#include <Arduino.h>
#include "mbed.h"

// ======== ユーザー設定 ========
#define RES_BITS         16   // 12 / 14 / 16 / 18
#define CHANNEL          1    // 1..4
#define PGA_GAIN         1    // 1,2,4,8
#define I2C_CLOCK_HZ     100000

#define PIN_SDA_GPIO     2    // GP2
#define PIN_SCL_GPIO     3    // GP3
#define PIN_SQUARE       0    // GP0: 325 Hz 矩形
#define PIN_SINEPWM      1    // GP1: PWM正弦 (RCでアナログ化)

const uint8_t MCP3424_ADDR_7BIT = 0x68;
const float   VREF = 2.048f;

// MCP3424 I2C (mbed::I2Cは8bitアドレス渡し)
mbed::I2C i2c(digitalPinToPinName(PIN_SDA_GPIO),
              digitalPinToPinName(PIN_SCL_GPIO));

// ===== MCP3424ユーティリティ =====
static inline uint8_t pgaBitsFromGain(int gain) {
  switch (gain) {
    case 1:  return 0b00;
    case 2:  return 0b01;
    case 4:  return 0b10;
    case 8:  return 0b11;
    default: return 0b00;
  }
}

static inline uint8_t srBitsFromRes(int resBits) {
  // [3:2]=SR: 00=12bit(240sps), 01=14bit(60sps),
  //         10=16bit(15sps),   11=18bit(3.75sps)
  switch (resBits) {
    case 12: return 0b00 << 2;
    case 14: return 0b01 << 2;
    case 16: return 0b10 << 2;
    case 18: return 0b11 << 2;
    default: return 0b10 << 2;
  }
}

static inline float spsFromRes(int resBits) {
  switch (resBits) {
    case 12: return 240.0f;
    case 14: return 60.0f;
    case 16: return 15.0f;
    case 18: return 3.75f;
    default: return 15.0f;
  }
}

// READ_PERIODはおおよその周期。実測はRDY確認でOKにしてる
static inline uint32_t readPeriodUsFromRes(int resBits) {
  float sps = spsFromRes(resBits);
  uint32_t base = (uint32_t)(1000000.0f / sps);
  return base + 2000;
}

static inline uint8_t chBitsFromChannel(uint8_t ch1to4) {
  switch (ch1to4) {
    case 1: default: return (0b00 << 5);
    case 2: return (0b01 << 5);
    case 3: return (0b10 << 5);
    case 4: return (0b11 << 5);
  }
}

static inline uint8_t makeConfig(uint8_t ch, int resBits, int pgaGain) {
  const uint8_t MODE_CONT = 1 << 4;
  return (uint8_t)( chBitsFromChannel(ch)
                  | MODE_CONT
                  | srBitsFromRes(resBits)
                  | pgaBitsFromGain(pgaGain) );
}

bool mcp3424_writeConfig(uint8_t cfg) {
  char data[1] = { (char)cfg };
  int addr8 = (MCP3424_ADDR_7BIT << 1);  // 8bit address
  return (i2c.write(addr8, data, 1, false) == 0);
}

bool mcp3424_read_16orLess(int16_t &raw16, uint8_t &cfg_out) {
  char buf[3];
  int addr8 = (MCP3424_ADDR_7BIT << 1);
  if (i2c.read(addr8, buf, 3, false) != 0) return false;

  cfg_out = (uint8_t)buf[2];
  if (cfg_out & 0x80) return false; // RDY=1→変換中

  uint8_t msb = (uint8_t)buf[0];
  uint8_t lsb = (uint8_t)buf[1];
  raw16 = (int16_t)((msb << 8) | lsb); // 2の補数
  return true;
}

bool mcp3424_read_18(int32_t &raw18, uint8_t &cfg_out) {
  char buf[4];
  int addr8 = (MCP3424_ADDR_7BIT << 1);
  if (i2c.read(addr8, buf, 4, false) != 0) return false;

  cfg_out = (uint8_t)buf[3];
  if (cfg_out & 0x80) return false; // RDY=1→変換中

  int32_t val = ((uint32_t)(uint8_t)buf[0] << 16)
              | ((uint32_t)(uint8_t)buf[1] << 8)
              |  (uint8_t)buf[2];

  // 18bit符号拡張 (bit17が符号)
  if (val & 0x20000) {
    val |= 0xFFFC0000;
  }
  raw18 = val;
  return true;
}

static inline float rawToVolt(int32_t raw, int resBits, int pgaGain) {
  float denom;
  switch (resBits) {
    case 12: denom = 2048.0f;   break;
    case 14: denom = 8192.0f;   break;
    case 16: denom = 32768.0f;  break;
    case 18: denom = 131072.0f; break;
    default: denom = 32768.0f;  break;
  }
  return (raw * (VREF / denom)) / (float)pgaGain;
}

// ======== 波形生成 (DDS方式) ========
//
// コンセプト:
//   ・割り込みレートは 10 kHz (Tickerで100usごと)
//   ・位相アキュムレータを32bit回す
//   ・1ステップで進む位相量 = F_OUT / FS * 2^32
//     → 平均周波数はF_OUTに一致
//   ・sineTable[ 上位ビット ] でサインDUTYを得る
//   ・squareOutは sin >=0 の時HIGHみたいにして
//     325Hz 50%dutyを同期生成
//
// メリット:
//   ・IRQは10 kHzだけ。83 kHzより圧倒的に軽い
//   ・main loop と USB Serial が動ける

static const float F_OUT = 325.0f;     // 目標周波数
static const float FS_DDS = 10000.0f;  // DDS更新レート(Hz) = 割り込み周期^-1

// サイン波テーブル: 256ステップ (0..255)
#define SINE_TABLE_SIZE 256
volatile uint8_t sineTable[SINE_TABLE_SIZE];

// 位相アキュムレータ (32bit固定小数)
volatile uint32_t phaseAcc = 0;
uint32_t phaseStep = 0;

// mbedのTicker (10 kHzでISR)
mbed::Ticker waveTicker;

// 出力ピン (mbed低レベルで握る)
mbed::PwmOut    pwmOut(digitalPinToPinName(PIN_SINEPWM));
mbed::DigitalOut squareOut(digitalPinToPinName(PIN_SQUARE));

// サインテーブル生成 (0..255 → duty)
void initSineTable() {
  for (uint16_t i = 0; i < SINE_TABLE_SIZE; i++) {
    float s = sinf(2.0f * 3.14159265359f * (float)i / (float)SINE_TABLE_SIZE);
    float u = 0.5f * (s + 1.0f);           // 0.0〜1.0
    uint8_t duty = (uint8_t)(u * 255.0f + 0.5f);
    sineTable[i] = duty;
  }
}

// 割り込み: 10kHz周期
void waveISR() {
  // 位相を進める
  phaseAcc += phaseStep;

  // 上位8bitをテーブルindexにする（32bit→上位8bitで256分解能）
  uint8_t idx = (uint8_t)(phaseAcc >> 24); // [31:24]をindexに

  // 正弦PWM duty更新
  uint8_t duty8 = sineTable[idx];
  float duty01 = (float)duty8 / 255.0f;
  pwmOut.write(duty01);

  // 同期スクエア波 (sin>=0 <=> idx<128?)
  // サインテーブルは0..127が上側(正)、128..255が下側(負)という形で作ってあるので
  if (idx < 128) {
    squareOut = 1;
  } else {
    squareOut = 0;
  }
}

// ======== setup / loop ========
void setup() {
  Serial.begin(115200);
  while (!Serial) { }

  // I2C設定
  i2c.frequency(I2C_CLOCK_HZ);

  // MCP3424 連続変換モード開始
  uint8_t cfg = makeConfig(CHANNEL, RES_BITS, PGA_GAIN);
  if (!mcp3424_writeConfig(cfg)) {
    Serial.println(F("# MCP3424 config write failed"));
  }

  // CSVヘッダ
  Serial.println(F("# t_ms,raw,voltage_V"));

  // ===== 波形出力の初期化 =====
  // PWMキャリア周波数をセット
  // 100 kHz → period 10us
  pwmOut.period_us(10);
  pwmOut.write(0.5f); // とりあえず50%
  squareOut = 0;      // 初期はLOW

  // Sineテーブル用意
  initSineTable();

  // DDSステップ計算
  // phaseStep = F_OUT / FS_DDS * 2^32
  // 2^32 = 4294967296.0f
  phaseStep = (uint32_t)((F_OUT / FS_DDS) * 4294967296.0f);

  // 10 kHzでwaveISRを呼ぶ (100us)
  waveTicker.attach_us(mbed::callback(waveISR), 100);
}

void loop() {
  static const uint32_t READ_PERIOD_US = readPeriodUsFromRes(RES_BITS);
  static uint32_t nextTick = micros();

  uint32_t now = micros();
  if ((int32_t)(now - nextTick) >= 0) {
    nextTick += READ_PERIOD_US;

    uint8_t cfg;
    bool ok = false;

    if (RES_BITS == 18) {
      int32_t raw18 = 0;
      for (int i = 0; i < 6 && !ok; ++i) {
        ok = mcp3424_read_18(raw18, cfg);
        if (!ok) delayMicroseconds(5000);
      }
      if (ok) {
        float volt = rawToVolt(raw18, RES_BITS, PGA_GAIN);
        Serial.print(millis());
        Serial.print(',');
        Serial.print(raw18);
        Serial.print(',');
        Serial.println(volt, 7);
      }
    } else {
      int16_t raw16 = 0;
      for (int i = 0; i < 5 && !ok; ++i) {
        ok = mcp3424_read_16orLess(raw16, cfg);
        if (!ok) delayMicroseconds(2000);
      }
      if (ok) {
        float volt = rawToVolt((int32_t)raw16, RES_BITS, PGA_GAIN);
        Serial.print(millis());
        Serial.print(',');
        Serial.print(raw16);
        Serial.print(',');
        Serial.println(volt, 6);
      }
    }
  }

  // 軽いスリープでCPUを休ませる＆USBに吐くチャンスをあげる
  uint32_t remain = (uint32_t)((int32_t)(nextTick - micros()) > 0 ? (nextTick - micros()) : 0);
  if (remain > 2000) {
    delay(1);
  } else if (remain > 0) {
    delayMicroseconds(remain);
  }
}
