#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
はかりモード & 個数はかりモード GUI
- Decimation + 移動平均 + OneEuro(誤差ブースト)
- 直線性補正：N点キャリブレーション（N次式通過補間：CAL点で必ず誤差0）
- RMSE/最大誤差の表示、点の追加/削除/並べ替えUI
- OneEuro の g/ADC ゲインは局所傾き（多点は多項式微分）で更新
- 最小表示丸め：0.05 など任意ステップで正しく丸め
- 「単重セット」クリック時：フィルタをリセットし、安定を待ってから自動保存
- 設定は scale_config.json に保存
- 追加：CSVログ（時刻, raw, decim, decim+MA, decim+OneEuro, gross[g], net[g]）
"""

import csv
import json
import math
import queue
import statistics
import threading
import time
from collections import deque
from dataclasses import dataclass, field
from datetime import datetime
from decimal import Decimal, ROUND_HALF_UP
from pathlib import Path
from typing import Optional, Deque, List, Tuple

import tkinter as tk
from tkinter import ttk, messagebox

try:
    import serial
    from serial.tools import list_ports
except Exception:  # pragma: no cover
    serial = None
    list_ports = None

CONFIG_PATH = Path("scale_config.json")
LOG_DIR = Path("logs")
DISPLAY_STEPS = [0.001, 0.002, 0.005, 0.01, 0.02, 0.05, 0.1, 0.2, 0.5, 1.0, 2.0, 5.0, 10.0]


# ==== 任意ステップ丸め（0.05も正しく丸める） ====
def round_to_step_str(value: float, step: float) -> str:
    q = Decimal(str(step))
    dval = Decimal(str(value))
    n = (dval / q).quantize(Decimal("1"), rounding=ROUND_HALF_UP)
    v = (n * q).quantize(q, rounding=ROUND_HALF_UP)
    digits = max(0, -q.as_tuple().exponent)
    fmt = f"{{0:.{digits}f}}"
    return fmt.format(v)


# ==== OneEuro フィルタ（誤差ブースト付き） ====
class LowPass:
    def __init__(self):
        self._y: Optional[float] = None
        self._s: bool = False

    def apply(self, x: float, alpha: float) -> float:
        if not self._s:
            self._y = x
            self._s = True
            return x
        self._y = alpha * x + (1.0 - alpha) * self._y
        return self._y

    @property
    def last(self) -> Optional[float]:
        return self._y


def _alpha(dt: float, cutoff: float) -> float:
    tau = 1.0 / (2.0 * math.pi * max(1e-6, cutoff))
    return 1.0 / (1.0 + tau / max(1e-6, dt))


class OneEuroFilter:
    """
    fc = min_cutoff + beta*|dx_hat| + err_boost_k*max(|e|-err_eps_g, 0)  （上限 fc_max）
    e = x - y_prev  （可能なら g 単位で評価）
    """
    def __init__(
        self,
        min_cutoff: float = 0.01,
        beta: float = 0.005,
        d_cutoff: float = 1.0,
        err_boost_k: float = 1.0,
        err_eps_g: float = 0.5,
        fc_max: float = 10.0,
    ):
        self.min_cutoff = float(min_cutoff)
        self.beta = float(beta)
        self.d_cutoff = float(d_cutoff)
        self.err_boost_k = float(err_boost_k)
        self.err_eps_g = float(err_eps_g)
        self.fc_max = float(fc_max)
        self._x_prev: Optional[float] = None
        self._t_prev: Optional[float] = None
        self._dx_lp = LowPass()
        self._x_lp = LowPass()
        self._g_per_adc: Optional[float] = None  # 校正後に設定

    def set_gain_g_per_adc(self, g_per_adc: Optional[float]):
        self._g_per_adc = g_per_adc

    def reset(self):
        self._x_prev = None
        self._t_prev = None
        self._dx_lp = LowPass()
        self._x_lp = LowPass()

    def filter(self, x: float, t: float) -> float:
        if self._t_prev is None:
            self._t_prev, self._x_prev = t, x
            self._dx_lp.apply(0.0, 1.0)
            return self._x_lp.apply(x, 1.0)

        dt = max(1e-6, t - self._t_prev)
        dx = (x - (self._x_prev if self._x_prev is not None else x)) / dt

        alpha_d = _alpha(dt, self.d_cutoff)
        dx_hat = self._dx_lp.apply(dx, alpha_d)

        y_prev = self._x_lp.last if self._x_lp.last is not None else x
        err_adc = x - y_prev
        if self._g_per_adc:
            err_g = abs(err_adc) * abs(self._g_per_adc)
        else:
            err_g = abs(err_adc)

        fc = self.min_cutoff + self.beta * abs(dx_hat) + self.err_boost_k * max(err_g - self.err_eps_g, 0.0)
        fc = min(self.fc_max, max(1e-6, fc))
        alpha_c = _alpha(dt, fc)

        y = self._x_lp.apply(x, alpha_c)
        self._t_prev, self._x_prev = t, x
        return y


# ==== 多項式通過補間（N次式）ヘルパ ====
def _solve_linear_system(a: List[List[float]], b: List[float]) -> Optional[List[float]]:
    n = len(b)
    M = [row[:] + [b[i]] for i, row in enumerate(a)]
    for col in range(n):
        pivot = max(range(col, n), key=lambda r: abs(M[r][col]))
        if abs(M[pivot][col]) < 1e-12:
            return None
        if pivot != col:
            M[col], M[pivot] = M[pivot], M[col]
        piv = M[col][col]
        inv = 1.0 / piv
        for j in range(col, n + 1):
            M[col][j] *= inv
        for r in range(n):
            if r == col:
                continue
            factor = M[r][col]
            if factor == 0.0:
                continue
            for j in range(col, n + 1):
                M[r][j] -= factor * M[col][j]
    return [M[i][n] for i in range(n)]


def _polyfit_through_points(xs: List[float], ys: List[float]) -> Optional[List[float]]:
    k = len(xs)
    if k == 0:
        return None
    if k == 1:
        return [ys[0]]
    A = [[(xs[i] ** j) for j in range(k)] for i in range(k)]
    return _solve_linear_system(A, ys)


def _poly_eval(coeffs: List[float], x: float) -> float:
    y = 0.0
    for c in reversed(coeffs):
        y = y * x + c
    return y


def _poly_derivative_eval(coeffs: List[float], x: float) -> float:
    if len(coeffs) <= 1:
        return 0.0
    dy = 0.0
    for power in range(len(coeffs) - 1, 0, -1):
        c = coeffs[power]
        dy = dy * x + power * c
    return dy


# ==== 校正/設定データ構造 ====
@dataclass
class Calibration:
    zero_adc: Optional[float] = None
    span_adc: Optional[float] = None
    span_weight_g: Optional[float] = None
    points: List[Tuple[float, float]] = field(default_factory=list)

    _poly_coeffs_cache: Optional[List[float]] = field(default=None, init=False, repr=False)
    _poly_src_len: int = field(default=0, init=False, repr=False)

    def _invalidate(self):
        self._poly_coeffs_cache = None
        self._poly_src_len = len(self.points)

    def is_ready(self) -> bool:
        return (
            self.zero_adc is not None
            and self.span_adc is not None
            and self.span_weight_g is not None
            and (self.span_adc - self.zero_adc) != 0
        )

    def has_npoint(self) -> bool:
        return len(self.points) >= 2

    def _sorted_points(self) -> List[Tuple[float, float]]:
        return sorted(self.points, key=lambda p: p[0])

    def _ensure_poly(self) -> Optional[List[float]]:
        if not self.has_npoint():
            return None
        if self._poly_coeffs_cache is not None and self._poly_src_len == len(self.points):
            return self._poly_coeffs_cache
        pts = self._sorted_points()
        xs = [p[0] for p in pts]
        ys = [p[1] for p in pts]
        coeffs = _polyfit_through_points(xs, ys)
        self._poly_coeffs_cache = coeffs
        self._poly_src_len = len(self.points)
        return coeffs

    def adc_to_gram(self, adc: float) -> Optional[float]:
        coeffs = self._ensure_poly()
        if coeffs is not None:
            return _poly_eval(coeffs, adc)
        if not self.is_ready():
            return None
        return (adc - self.zero_adc) * (self.span_weight_g / (self.span_adc - self.zero_adc))

    def gain_at_adc(self, adc: Optional[float] = None) -> Optional[float]:
        coeffs = self._ensure_poly()
        if coeffs is not None:
            x = self._sorted_points()[0][0] if adc is None else adc
            return _poly_derivative_eval(coeffs, x)
        if self.is_ready():
            return self.span_weight_g / (self.span_adc - self.zero_adc)
        return None

    def stats(self) -> Optional[Tuple[float, float]]:
        if not self.has_npoint():
            return None
        pts = self._sorted_points()
        errs = []
        for x, y_true in pts:
            y_hat = self.adc_to_gram(x)
            if y_hat is None:
                continue
            errs.append(y_hat - y_true)
        if not errs:
            return None
        rmse = math.sqrt(sum(e * e for e in errs) / len(errs))
        max_abs = max(abs(e) for e in errs)
        return (rmse, max_abs)


@dataclass
class ScaleConfig:
    port: str = ""
    baudrate: int = 115200
    display_step_g: float = 0.1
    ma_window: int = 15
    decim_factor: int = 4
    apply_ma_to_display: bool = True
    stability_std_threshold_g: float = 0.02
    calibration: Calibration = field(default_factory=Calibration)
    unit_weight_g: Optional[float] = None

    # OneEuro 既定値（ご指定）
    use_one_euro: bool = True
    one_euro_min_cutoff: float = 0.01
    one_euro_beta: float = 0.005
    one_euro_d_cutoff: float = 1.0
    one_euro_err_boost_k: float = 1.0
    one_euro_err_eps_g: float = 0.5
    one_euro_fc_max: float = 10.0

    @classmethod
    def load(cls) -> "ScaleConfig":
        if CONFIG_PATH.exists():
            try:
                data = json.loads(CONFIG_PATH.read_text("utf-8"))
                cal = data.get("calibration", {})
                return cls(
                    port=data.get("port", ""),
                    baudrate=int(data.get("baudrate", 115200)),
                    display_step_g=float(data.get("display_step_g", 0.1)),
                    ma_window=int(data.get("ma_window", 15)),
                    decim_factor=int(data.get("decim_factor", 4)),
                    apply_ma_to_display=bool(data.get("apply_ma_to_display", True)),
                    stability_std_threshold_g=float(data.get("stability_std_threshold_g", 0.02)),
                    calibration=Calibration(
                        zero_adc=cal.get("zero_adc"),
                        span_adc=cal.get("span_adc"),
                        span_weight_g=cal.get("span_weight_g"),
                        points=[tuple(p) for p in cal.get("points", [])],
                    ),
                    unit_weight_g=(data.get("unit_weight_g")),
                    use_one_euro=bool(data.get("use_one_euro", True)),
                    one_euro_min_cutoff=float(data.get("one_euro_min_cutoff", 0.01)),
                    one_euro_beta=float(data.get("one_euro_beta", 0.005)),
                    one_euro_d_cutoff=float(data.get("one_euro_d_cutoff", 1.0)),
                    one_euro_err_boost_k=float(data.get("one_euro_err_boost_k", 1.0)),
                    one_euro_err_eps_g=float(data.get("one_euro_err_eps_g", 0.5)),
                    one_euro_fc_max=float(data.get("one_euro_fc_max", 10.0)),
                )
            except Exception:
                pass
        return cls()

    def save(self) -> None:
        data = {
            "port": self.port,
            "baudrate": self.baudrate,
            "display_step_g": self.display_step_g,
            "ma_window": self.ma_window,
            "decim_factor": self.decim_factor,
            "apply_ma_to_display": self.apply_ma_to_display,
            "stability_std_threshold_g": self.stability_std_threshold_g,
            "calibration": {
                "zero_adc": self.calibration.zero_adc,
                "span_adc": self.calibration.span_adc,
                "span_weight_g": self.calibration.span_weight_g,
                "points": self.calibration.points,
            },
            "unit_weight_g": self.unit_weight_g,
            "use_one_euro": self.use_one_euro,
            "one_euro_min_cutoff": self.one_euro_min_cutoff,
            "one_euro_beta": self.one_euro_beta,
            "one_euro_d_cutoff": self.one_euro_d_cutoff,
            "one_euro_err_boost_k": self.one_euro_err_boost_k,
            "one_euro_err_eps_g": self.one_euro_err_eps_g,
            "one_euro_fc_max": self.one_euro_fc_max,
        }
        CONFIG_PATH.write_text(json.dumps(data, ensure_ascii=False, indent=2), "utf-8")


# ==== シリアル受信 ====
class SerialReader(threading.Thread):
    def __init__(self, port: str, baud: int, out_queue: queue.Queue):
        super().__init__(daemon=True)
        self.port = port
        self.baud = baud
        self.out_queue = out_queue
        self._stop_event = threading.Event()
        self._ser = None

    def run(self):
        try:
            self._ser = serial.Serial(self.port, self.baud, timeout=1)
        except Exception as e:  # pragma: no cover
            self.out_queue.put(("__error__", f"ポートを開けませんでした: {e}"))
            return

        buf = bytearray()
        while not self._stop_event.is_set():
            try:
                ch = self._ser.read(1)
                if not ch:
                    continue
                c = ch[0]
                if c in (10, 13):  # LF/CR
                    if buf:
                        line = buf.decode("utf-8", errors="ignore").strip()
                        buf.clear()
                        self._handle_line(line)
                else:
                    buf.append(c)
            except Exception as e:  # pragma: no cover
                self.out_queue.put(("__error__", f"受信エラー: {e}"))
                time.sleep(0.2)
                break

        try:
            if self._ser and self._ser.is_open:
                self._ser.close()
        except Exception:
            pass

    def _handle_line(self, line: str):
        try:
            cols = line.split(",")
            if len(cols) >= 2:
                adc = float(cols[1])
                self.out_queue.put(("adc", adc))
        except Exception:
            pass

    def stop(self):
        self._stop_event.set()


# ==== GUI本体 ====
class App(tk.Tk):
    def __init__(self):
        super().__init__()
        self.title("はかり & 個数はかり GUI (Decim + MA + OneEuro + N次式補正)")
        self.geometry("1180x850")
        self.minsize(1100, 780)

        self.config = ScaleConfig.load()
        self.reader: Optional[SerialReader] = None
        self.q: queue.Queue = queue.Queue()

        # シリーズ
        self.adc_window_raw: Deque[float] = deque(maxlen=2000)
        self.adc_window_decim: Deque[float] = deque(maxlen=2000)
        self.adc_window_decim_ma: Deque[float] = deque(maxlen=2000)
        self.adc_window_decim_euro: Deque[float] = deque(maxlen=2000)

        # デシメーション状態
        self._decim_sum = 0.0
        self._decim_count = 0

        # OneEuro 状態
        self._euro = OneEuroFilter(
            self.config.one_euro_min_cutoff,
            self.config.one_euro_beta,
            self.config.one_euro_d_cutoff,
            self.config.one_euro_err_boost_k,
            self.config.one_euro_err_eps_g,
            self.config.one_euro_fc_max,
        )

        self._net_tare_g: float = 0.0

        # 単重セットの安定待ち状態
        self._uw_pending_n: Optional[int] = None
        self._uw_dwell_start: Optional[float] = None
        self._uw_dwell_ms: int = 400

        # ログ関連
        self.log_enabled: bool = False
        self.log_fp = None
        self.log_writer: Optional[csv.writer] = None
        self.var_logfile = tk.StringVar(value="ログ: 停止中")

        self._build_ui()
        self._apply_config_to_ui()
        self._update_euro_gain()
        self.after(40, self._poll_queue)

    # ===== UI =====
    def _build_ui(self):
        frm_top = ttk.LabelFrame(self, text="シリアル接続")
        frm_top.pack(fill=tk.X, padx=10, pady=8)

        ttk.Label(frm_top, text="ポート").grid(row=0, column=0, padx=6, pady=6)
        self.cmb_port = ttk.Combobox(frm_top, width=26, state="readonly")
        self.cmb_port.grid(row=0, column=1, padx=6, pady=6)
        ttk.Button(frm_top, text="更新", command=self._refresh_ports).grid(row=0, column=2, padx=6, pady=6)

        ttk.Label(frm_top, text="ボーレート").grid(row=0, column=3, padx=6, pady=6)
        self.ent_baud = ttk.Entry(frm_top, width=10)
        self.ent_baud.grid(row=0, column=4, padx=6, pady=6)

        self.btn_connect = ttk.Button(frm_top, text="接続", command=self._connect)
        self.btn_connect.grid(row=0, column=5, padx=6, pady=6)
        self.btn_disconnect = ttk.Button(frm_top, text="切断", command=self._disconnect, state=tk.DISABLED)
        self.btn_disconnect.grid(row=0, column=6, padx=6, pady=6)

        # ログ操作
        ttk.Button(frm_top, text="ログ開始", command=self._start_logging).grid(row=0, column=7, padx=6, pady=6)
        ttk.Button(frm_top, text="ログ停止", command=self._stop_logging).grid(row=0, column=8, padx=6, pady=6)
        ttk.Label(frm_top, textvariable=self.var_logfile, width=40).grid(row=0, column=9, padx=6, pady=6, sticky=tk.W)

        frm_disp = ttk.LabelFrame(self, text="現在値（Decimation + MA / OneEuro）")
        frm_disp.pack(fill=tk.X, padx=10, pady=8)

        self.var_status = tk.StringVar(value="未接続")
        ttk.Label(frm_disp, textvariable=self.var_status, width=56).grid(row=0, column=0, padx=6, pady=6, sticky=tk.W)

        ttk.Label(frm_disp, text="最小表示 (g)").grid(row=0, column=1, padx=6, pady=6)
        self.cmb_step = ttk.Combobox(frm_disp, width=8, state="readonly", values=[str(s) for s in DISPLAY_STEPS])
        self.cmb_step.grid(row=0, column=2, padx=6, pady=6)
        self.cmb_step.bind("<<ComboboxSelected>>", lambda e: self._on_step_changed())

        ttk.Label(frm_disp, text="Decim D").grid(row=0, column=3, padx=6, pady=6)
        self.spn_decim = tk.Spinbox(frm_disp, from_=1, to=128, width=6)
        self.spn_decim.grid(row=0, column=4, padx=6, pady=6)
        self.spn_decim.bind("<Return>", lambda e: self._on_decim_changed())
        self.spn_decim.bind("<FocusOut>", lambda e: self._on_decim_changed())

        ttk.Label(frm_disp, text="MA N").grid(row=0, column=5, padx=6, pady=6)
        self.spn_ma = tk.Spinbox(frm_disp, from_=1, to=500, width=6)
        self.spn_ma.grid(row=0, column=6, padx=6, pady=6)
        self.spn_ma.bind("<Return>", lambda e: self._on_ma_changed())
        self.spn_ma.bind("<FocusOut>", lambda e: self._on_ma_changed())

        self.var_apply_ma = tk.BooleanVar(value=True)
        self.chk_apply_ma = ttk.Checkbutton(
            frm_disp,
            text="表示に移動平均を適用（OFF=Decimのみ）",
            variable=self.var_apply_ma,
            command=self._on_apply_ma_changed,
        )
        self.chk_apply_ma.grid(row=0, column=7, padx=6, pady=6)

        ttk.Label(frm_disp, text="安定判定 σ しきい値 (g)").grid(row=0, column=8, padx=6, pady=6)
        self.ent_std_th = ttk.Entry(frm_disp, width=8)
        self.ent_std_th.grid(row=0, column=9, padx=6, pady=6)

        self.stable_led = tk.Canvas(frm_disp, width=18, height=18, highlightthickness=0)
        self.stable_led.grid(row=0, column=10, padx=6, pady=6)
        self._draw_led(self.stable_led, False)

        # 数値表示
        frm_vals = ttk.Frame(frm_disp)
        frm_vals.grid(row=1, column=0, columnspan=12, sticky=tk.W, padx=6, pady=4)

        self.var_adc_raw = tk.StringVar(value="ADC(raw): ---")
        self.var_adc_dec = tk.StringVar(value="ADC(decim): ---")
        self.var_adc_ma = tk.StringVar(value="ADC(decim+MA): ---")
        self.var_adc_eu = tk.StringVar(value="ADC(decim+OneEuro): ---")

        self.var_gross = tk.StringVar(value="総重量(g): ---")
        self.var_net = tk.StringVar(value="ネット(g): ---")

        ttk.Label(frm_vals, textvariable=self.var_adc_raw, width=30).grid(row=0, column=0, padx=6, pady=2, sticky=tk.W)
        ttk.Label(frm_vals, textvariable=self.var_adc_dec, width=30).grid(row=0, column=1, padx=6, pady=2, sticky=tk.W)
        ttk.Label(frm_vals, textvariable=self.var_adc_ma, width=32).grid(row=0, column=2, padx=6, pady=2, sticky=tk.W)
        ttk.Label(frm_vals, textvariable=self.var_adc_eu, width=34).grid(row=0, column=3, padx=6, pady=2, sticky=tk.W)

        ttk.Label(frm_vals, textvariable=self.var_gross, font=("Segoe UI", 32, "bold"), foreground="blue").grid(
            row=1, column=0, columnspan=3, padx=6, pady=8, sticky=tk.W
        )
        ttk.Label(frm_vals, textvariable=self.var_net, font=("Segoe UI", 32, "bold"), foreground="green").grid(
            row=2, column=0, columnspan=3, padx=6, pady=8, sticky=tk.W
        )

        # === OneEuro 設定 ===
        frm_eu = ttk.LabelFrame(self, text="One Euro Filter（誤差ブースト付き）")
        frm_eu.pack(fill=tk.X, padx=10, pady=4)

        self.var_use_eu = tk.BooleanVar(value=self.config.use_one_euro)
        self.chk_use_eu = ttk.Checkbutton(
            frm_eu,
            text="OneEuro を有効化（表示/安定判定に Decim+OneEuro を使用）",
            variable=self.var_use_eu,
            command=self._on_toggle_euro,
        )
        self.chk_use_eu.grid(row=0, column=0, padx=6, pady=6, sticky=tk.W)

        ttk.Label(frm_eu, text="min_cutoff [Hz]").grid(row=0, column=1, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_min = ttk.Entry(frm_eu, width=8)
        self.ent_eu_min.grid(row=0, column=2, padx=4, pady=6)

        ttk.Label(frm_eu, text="beta").grid(row=0, column=3, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_beta = ttk.Entry(frm_eu, width=8)
        self.ent_eu_beta.grid(row=0, column=4, padx=4, pady=6)

        ttk.Label(frm_eu, text="d_cutoff [Hz]").grid(row=0, column=5, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_d = ttk.Entry(frm_eu, width=8)
        self.ent_eu_d.grid(row=0, column=6, padx=4, pady=6)

        ttk.Label(frm_eu, text="err_boost_k").grid(row=1, column=1, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_errk = ttk.Entry(frm_eu, width=8)
        self.ent_eu_errk.grid(row=1, column=2, padx=4, pady=6)

        ttk.Label(frm_eu, text="err_eps_g [g]").grid(row=1, column=3, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_err_eps = ttk.Entry(frm_eu, width=8)
        self.ent_eu_err_eps.grid(row=1, column=4, padx=4, pady=6)

        ttk.Label(frm_eu, text="fc_max [Hz]").grid(row=1, column=5, padx=6, pady=6, sticky=tk.E)
        self.ent_eu_fcmax = ttk.Entry(frm_eu, width=8)
        self.ent_eu_fcmax.grid(row=1, column=6, padx=4, pady=6)

        ttk.Button(frm_eu, text="適用", command=self._apply_euro_params).grid(
            row=0, column=7, rowspan=2, padx=10, pady=6, sticky=tk.NS
        )

        # Notebook
        self.nb = ttk.Notebook(self)
        self.nb.pack(fill=tk.BOTH, expand=True, padx=10, pady=4)

        self.tab_scale = ttk.Frame(self.nb)
        self.nb.add(self.tab_scale, text="はかりモード")
        self._build_tab_scale(self.tab_scale)

        self.tab_count = ttk.Frame(self.nb)
        self.nb.add(self.tab_count, text="個数はかりモード")
        self._build_tab_count(self.tab_count)

        self.tab_caln = ttk.Frame(self.nb)
        self.nb.add(self.tab_caln, text="多点CAL（N次式補正）")
        self._build_tab_caln(self.tab_caln)

    def _build_tab_scale(self, parent: ttk.Frame):
        frm = ttk.Frame(parent)
        frm.pack(fill=tk.X, padx=10, pady=10)

        cal_box = ttk.LabelFrame(frm, text="CAL（2点校正）")
        cal_box.grid(row=0, column=0, padx=6, pady=6, sticky=tk.W)
        ttk.Button(cal_box, text="ゼロCAL (荷重なしで押す)", command=self._cal_zero).grid(
            row=0, column=0, padx=6, pady=6, sticky=tk.W
        )
        ttk.Label(cal_box, text="既知重量(g)").grid(row=1, column=0, padx=6, pady=6, sticky=tk.W)
        self.ent_span_w = ttk.Entry(cal_box, width=10)
        self.ent_span_w.grid(row=1, column=1, padx=6, pady=6)
        ttk.Button(cal_box, text="スパンCAL (既知荷重を載せて)", command=self._cal_span).grid(
            row=1, column=2, padx=6, pady=6
        )
        ttk.Button(cal_box, text="CALリセット", command=self._cal_reset).grid(row=2, column=0, padx=6, pady=6)

        zero_box = ttk.LabelFrame(frm, text="ゼロ引き（風袋）")
        zero_box.grid(row=0, column=1, padx=16, pady=6, sticky=tk.W)
        ttk.Button(zero_box, text="ゼロ引き", command=self._tare_now).grid(row=0, column=0, padx=6, pady=6)
        ttk.Button(zero_box, text="ゼロ解除", command=self._tare_clear).grid(row=0, column=1, padx=6, pady=6)

        info_box = ttk.LabelFrame(frm, text="校正情報")
        info_box.grid(row=0, column=2, padx=6, pady=6, sticky=tk.W)
        self.var_cal_info = tk.StringVar(value="未校正")
        ttk.Label(info_box, textvariable=self.var_cal_info, width=56).grid(row=0, column=0, padx=6, pady=6)

    def _build_tab_count(self, parent: ttk.Frame):
        frm = ttk.Frame(parent)
        frm.pack(fill=tk.X, padx=10, pady=10)

        uw_box = ttk.LabelFrame(frm, text="単重セット")
        uw_box.grid(row=0, column=0, padx=6, pady=6, sticky=tk.W)
        ttk.Label(uw_box, text="サンプル個数 N").grid(row=0, column=0, padx=6, pady=6)
        self.ent_sample_n = ttk.Entry(uw_box, width=8)
        self.ent_sample_n.grid(row=0, column=1, padx=6, pady=6)
        ttk.Button(uw_box, text="安定待ち→単重セット", command=self._start_unit_weight_capture).grid(
            row=0, column=2, padx=6, pady=6
        )
        ttk.Button(uw_box, text="単重クリア", command=self._clear_unit_weight).grid(row=0, column=3, padx=6, pady=6)
        self.var_unit_weight = tk.StringVar(value="単重(g): ---")
        ttk.Label(uw_box, textvariable=self.var_unit_weight, width=28).grid(
            row=1, column=0, columnspan=3, padx=6, pady=6, sticky=tk.W
        )

        zero_box = ttk.LabelFrame(frm, text="ゼロ引き（風袋）")
        zero_box.grid(row=0, column=1, padx=16, pady=6, sticky=tk.W)
        ttk.Button(zero_box, text="ゼロ引き", command=self._tare_now).grid(row=0, column=0, padx=6, pady=6)
        ttk.Button(zero_box, text="ゼロ解除", command=self._tare_clear).grid(row=0, column=1, padx=6, pady=6)

        disp_box = ttk.LabelFrame(frm, text="個数表示")
        disp_box.grid(row=1, column=0, columnspan=2, padx=6, pady=12, sticky=tk.W)
        self.var_count = tk.StringVar(value="--- 個")
        ttk.Label(disp_box, textvariable=self.var_count, font=("Segoe UI", 110, "bold")).grid(
            row=0, column=0, padx=8, pady=4
        )
        self.var_count_detail = tk.StringVar(value="詳細: ---")
        ttk.Label(disp_box, textvariable=self.var_count_detail).grid(row=1, column=0, padx=8, pady=4, sticky=tk.W)

    def _build_tab_caln(self, parent: ttk.Frame):
        frm = ttk.Frame(parent)
        frm.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)

        ctrl = ttk.LabelFrame(frm, text="キャリブレーション点（ADC→g：N次式通過）")
        ctrl.pack(fill=tk.X, padx=4, pady=4)
        ttk.Label(ctrl, text="既知重量(g)").grid(row=0, column=0, padx=6, pady=6, sticky=tk.E)
        self.ent_caln_w = ttk.Entry(ctrl, width=10)
        self.ent_caln_w.grid(row=0, column=1, padx=6, pady=6)
        ttk.Button(ctrl, text="現在値を追加", command=self._caln_add_current).grid(row=0, column=2, padx=6, pady=6)
        ttk.Button(ctrl, text="選択削除", command=self._caln_del_selected).grid(row=0, column=3, padx=6, pady=6)
        ttk.Button(ctrl, text="全クリア", command=self._caln_clear).grid(row=0, column=4, padx=6, pady=6)
        ttk.Button(ctrl, text="ADCで昇順ソート", command=self._caln_sort).grid(row=0, column=5, padx=6, pady=6)
        self.var_caln_stats = tk.StringVar(value="点数: 0, RMSE: -, MaxErr: -")
        ttk.Label(ctrl, textvariable=self.var_caln_stats).grid(row=0, column=6, padx=12, pady=6, sticky=tk.W)

        tbl = ttk.Treeview(frm, columns=("adc", "g"), show="headings", height=14)
        tbl.heading("adc", text="ADC")
        tbl.heading("g", text="重量[g]")
        tbl.column("adc", width=220, anchor=tk.E)
        tbl.column("g", width=140, anchor=tk.E)
        tbl.pack(fill=tk.BOTH, expand=True, padx=4, pady=6)
        self.tree_caln = tbl
        self._caln_refresh_table()

    # ===== 設定反映 =====
    def _apply_config_to_ui(self):
        self._refresh_ports()
        if self.config.port:
            try:
                self.cmb_port.set(self.config.port)
            except Exception:
                pass
        self.ent_baud.delete(0, tk.END)
        self.ent_baud.insert(0, str(self.config.baudrate))

        self.cmb_step.set(str(self.config.display_step_g))
        self.spn_decim.delete(0, tk.END)
        self.spn_decim.insert(0, str(self.config.decim_factor))
        self.spn_ma.delete(0, tk.END)
        self.spn_ma.insert(0, str(self.config.ma_window))
        self.var_apply_ma.set(self.config.apply_ma_to_display)
        self.ent_std_th.delete(0, tk.END)
        self.ent_std_th.insert(0, str(self.config.stability_std_threshold_g))

        self.var_use_eu.set(self.config.use_one_euro)
        self.ent_eu_min.delete(0, tk.END)
        self.ent_eu_min.insert(0, str(self.config.one_euro_min_cutoff))
        self.ent_eu_beta.delete(0, tk.END)
        self.ent_eu_beta.insert(0, str(self.config.one_euro_beta))
        self.ent_eu_d.delete(0, tk.END)
        self.ent_eu_d.insert(0, str(self.config.one_euro_d_cutoff))
        self.ent_eu_errk.delete(0, tk.END)
        self.ent_eu_errk.insert(0, str(self.config.one_euro_err_boost_k))
        self.ent_eu_err_eps.delete(0, tk.END)
        self.ent_eu_err_eps.insert(0, str(self.config.one_euro_err_eps_g))
        self.ent_eu_fcmax.delete(0, tk.END)
        self.ent_eu_fcmax.insert(0, str(self.config.one_euro_fc_max))

        self._update_cal_info_label()
        self._update_unit_weight_label()

    # ===== ログ =====
    def _start_logging(self):
        try:
            LOG_DIR.mkdir(parents=True, exist_ok=True)
            ts = datetime.now().strftime("%Y%m%d_%H%M%S")
            fpath = LOG_DIR / f"scale_{ts}.csv"
            self.log_fp = fpath.open("w", newline="", encoding="utf-8")
            self.log_writer = csv.writer(self.log_fp)
            self.log_writer.writerow(
                ["time_iso", "adc_raw", "adc_decim", "adc_decim_ma", "adc_decim_oneeuro", "gross_g", "net_g"]
            )
            self.log_enabled = True
            self.var_logfile.set(f"ログ: {fpath.name}")
            self.var_status.set("ログを開始しました")
        except Exception as e:
            messagebox.showerror("ログ開始エラー", str(e))

    def _stop_logging(self):
        try:
            if self.log_fp:
                self.log_fp.flush()
                self.log_fp.close()
        except Exception:
            pass
        self.log_fp = None
        self.log_writer = None
        self.log_enabled = False
        self.var_logfile.set("ログ: 停止中")
        self.var_status.set("ログを停止しました")

    def _log_sample(self, raw, dec, dma, deu, gross, net):
        if not self.log_enabled or self.log_writer is None:
            return
        t = datetime.now().strftime("%Y-%m-%dT%H:%M:%S.%f")[:-3]
        def fmt(v):
            return "" if v is None else f"{v:.6f}"
        self.log_writer.writerow([t, fmt(raw), fmt(dec), fmt(dma), fmt(deu), fmt(gross), fmt(net)])
        # 逐次 flush（安全優先）
        try:
            self.log_fp.flush()
        except Exception:
            pass

    # ===== ポート関連 =====
    def _refresh_ports(self):
        if list_ports is None:
            self.cmb_port["values"] = []
            return
        items = [f"{p.device} ({p.description})" for p in list_ports.comports()]
        self.cmb_port["values"] = items
        if items and not self.cmb_port.get():
            self.cmb_port.current(0)

    def _selected_port_name(self) -> Optional[str]:
        sel = self.cmb_port.get()
        if not sel:
            return None
        return sel.split(" ")[0]

    # ===== 接続/切断 =====
    def _connect(self):
        if serial is None:
            messagebox.showerror("エラー", "pyserial が未インストールです\npip install pyserial を実行してください")
            return
        port = self._selected_port_name()
        if not port:
            messagebox.showwarning("ポート未選択", "シリアルポートを選択してください")
            return
        try:
            baud = int(self.ent_baud.get())
        except ValueError:
            messagebox.showwarning("ボーレートエラー", "ボーレートは整数で入力してください")
            return
        self.config.port = port
        self.config.baudrate = baud
        self.config.save()
        self.reader = SerialReader(port, baud, self.q)
        self.reader.start()
        self.btn_connect.configure(state=tk.DISABLED)
        self.btn_disconnect.configure(state=tk.NORMAL)
        self.var_status.set(f"接続中: {port} @{baud}")

    def _disconnect(self):
        if self.reader is not None:
            self.reader.stop()
            self.reader.join(timeout=1.0)
            self.reader = None
        self.btn_connect.configure(state=tk.NORMAL)
        self.btn_disconnect.configure(state=tk.DISABLED)
        self.var_status.set("切断しました")

    # ===== 校正/ゼロ/単重 =====
    def _cal_zero(self):
        adc = self._choose_adc()
        if adc is None:
            messagebox.showinfo("情報", "受信中の安定した ADC 値が必要です")
            return
        self.config.calibration.zero_adc = adc
        self.config.calibration._invalidate()
        self.config.save()
        self._update_cal_info_label()
        self._update_euro_gain()
        messagebox.showinfo("ゼロCAL", f"ゼロADC = {adc:.6f} を保存しました")

    def _cal_span(self):
        adc = self._choose_adc()
        if adc is None:
            messagebox.showinfo("情報", "受信中の安定した ADC 値が必要です")
            return
        try:
            w = float(self.ent_span_w.get())
            if w <= 0:
                raise ValueError
        except Exception:
            messagebox.showwarning("入力エラー", "既知重量(g) を正しく入力してください")
            return
        self.config.calibration.span_adc = adc
        self.config.calibration.span_weight_g = w
        self.config.calibration._invalidate()
        self.config.save()
        self._update_cal_info_label()
        self._update_euro_gain()
        if self.config.calibration.is_ready():
            messagebox.showinfo("スパンCAL", f"スパンADC = {adc:.6f}, 既知重量 = {w} g → 校正完了")
        else:
            messagebox.showinfo("スパンCAL", f"スパンADC = {adc:.6f}, 既知重量 = {w} g を保存（ゼロCALも実施）")

    def _cal_reset(self):
        self.config.calibration = Calibration()
        self.config.save()
        self._update_cal_info_label()
        self._update_euro_gain()
        self._caln_refresh_table()
        messagebox.showinfo("CALリセット", "校正値をクリアしました")

    def _tare_now(self):
        gross = self._current_weight_gross()
        if gross is None:
            messagebox.showinfo("情報", "重量が未校正、または値が未取得です")
            return
        self._net_tare_g = gross
        self.var_status.set("ゼロ引き中（風袋を差し引き）")

    def _tare_clear(self):
        self._net_tare_g = 0.0
        self.var_status.set("ゼロ解除しました")

    # === 単重セット：安定待ちで自動保存 ===
    def _start_unit_weight_capture(self):
        try:
            n = int(self.ent_sample_n.get())
            if n <= 0:
                raise ValueError
        except Exception:
            messagebox.showwarning("入力エラー", "サンプル個数 N は 1 以上の整数で入力してください")
            return
        self._reset_filters()
        self._uw_pending_n = n
        self._uw_dwell_start = None
        self.var_status.set(f"単重セット: 安定待ち中（N={n}）…")

    def _finish_unit_weight_capture_if_ready(self, net: Optional[float]):
        if self._uw_pending_n is None:
            return
        if net is None or net <= 0:
            self._uw_dwell_start = None
            return
        if not self._is_stable():
            self._uw_dwell_start = None
            return
        now = time.monotonic()
        if self._uw_dwell_start is None:
            self._uw_dwell_start = now
            return
        dwell = (now - self._uw_dwell_start) * 1000.0
        if dwell >= self._uw_dwell_ms:
            uw = net / self._uw_pending_n
            self.config.unit_weight_g = uw
            self.config.save()
            self._update_unit_weight_label()
            messagebox.showinfo("単重セット", f"安定を検出 → 単重 = {uw:.6f} g/個 を保存しました")
            self.var_status.set("単重セット完了")
            self._uw_pending_n = None
            self._uw_dwell_start = None

    def _clear_unit_weight(self):
        self.config.unit_weight_g = None
        self.config.save()
        self._update_unit_weight_label()

    # ===== 受信処理 =====
    def _poll_queue(self):
        updated = False
        while True:
            try:
                kind, payload = self.q.get_nowait()
            except queue.Empty:
                break
            if kind == "adc":
                x = float(payload)
                self._push_raw(x)
                updated = True
            elif kind == "__error__":
                messagebox.showerror("受信エラー", str(payload))
                self._disconnect()
                break

        if updated:
            self._update_display()
        self.after(40, self._poll_queue)

    # ===== シリーズ投入（Raw→Decim→ MA/OneEuro） =====
    def _push_raw(self, x: float):
        # Raw
        self.adc_window_raw.append(x)

        # Decimation（D点平均）
        D = max(1, self._get_decim_factor())
        self._decim_sum += x
        self._decim_count += 1

        if self._decim_count >= D:
            avgD = self._decim_sum / self._decim_count
            self.adc_window_decim.append(avgD)
            self._decim_sum = 0.0
            self._decim_count = 0

            # Decim の移動平均（M点）
            M = max(1, self._get_ma_window())
            data = list(self.adc_window_decim)[-M:]
            avgDM = sum(data) / len(data)
            self.adc_window_decim_ma.append(avgDM)

            # Decim の OneEuro（適応）
            t_now = time.monotonic()
            y_eu = self._euro.filter(avgD, t_now)
            self.adc_window_decim_euro.append(y_eu)

            # ログ（デシメーション一回ごとに1行）
            raw = self._current_adc_raw()
            dec = self._current_adc_decim()
            dma = self._current_adc_decim_ma()
            deu = self._current_adc_decim_euro()
            gross = self._current_weight_gross()
            net = self._current_weight_net()
            self._log_sample(raw, dec, dma, deu, gross, net)

    # ===== 表示更新 =====
    def _update_display(self):
        raw = self._current_adc_raw()
        dec = self._current_adc_decim()
        dma = self._current_adc_decim_ma()
        deu = self._current_adc_decim_euro()

        self.var_adc_raw.set("ADC(raw): ---" if raw is None else f"ADC(raw): {raw:.6f}")
        self.var_adc_dec.set("ADC(decim): ---" if dec is None else f"ADC(decim): {dec:.6f}")
        self.var_adc_ma.set("ADC(decim+MA): ---" if dma is None else f"ADC(decim+MA): {dma:.6f}")
        self.var_adc_eu.set("ADC(decim+OneEuro): ---" if deu is None else f"ADC(decim+OneEuro): {deu:.6f}")

        gross = self._current_weight_gross()
        net = self._current_weight_net()
        step = float(self.cmb_step.get() or self.config.display_step_g)

        self.var_gross.set("総重量(g): 未校正" if gross is None else f"総重量(g): {round_to_step_str(gross, step)}")
        self.var_net.set("ネット(g): 未校正" if net is None else f"ネット(g): {round_to_step_str(net, step)}")

        self._finish_unit_weight_capture_if_ready(net)

        is_stable = self._is_stable()
        self._draw_led(self.stable_led, is_stable)

        self._update_euro_gain()

        if self.nb.index(self.nb.select()) == 1:
            self._update_count_display(net)

    def _update_count_display(self, net: Optional[float]):
        if self.config.unit_weight_g is None or net is None or self.config.unit_weight_g <= 0:
            self.var_count.set("--- 個")
            self.var_count_detail.set("詳細: 単重未設定 or 未校正")
            return
        pieces_float = net / self.config.unit_weight_g
        pieces_int = int(max(0, round(pieces_float)))
        self.var_count.set(f"{pieces_int} 個")
        self.var_count_detail.set(
            f"net={net:.6f} g, uw={self.config.unit_weight_g:.6f} g/個 → {pieces_float:.3f} 個"
        )

    # ===== 内部計算 =====
    def _reset_filters(self):
        self.adc_window_raw.clear()
        self.adc_window_decim.clear()
        self.adc_window_decim_ma.clear()
        self.adc_window_decim_euro.clear()
        self._decim_sum = 0.0
        self._decim_count = 0
        self._euro.reset()

    def _current_adc_raw(self) -> Optional[float]:
        return self.adc_window_raw[-1] if self.adc_window_raw else None

    def _current_adc_decim(self) -> Optional[float]:
        return self.adc_window_decim[-1] if self.adc_window_decim else None

    def _current_adc_decim_ma(self) -> Optional[float]:
        return self.adc_window_decim_ma[-1] if self.adc_window_decim_ma else None

    def _current_adc_decim_euro(self) -> Optional[float]:
        return self.adc_window_decim_euro[-1] if self.adc_window_decim_euro else None

    def _choose_adc(self) -> Optional[float]:
        if self.config.use_one_euro:
            v = self._current_adc_decim_euro()
            if v is not None:
                return v
        if self.var_apply_ma.get():
            v = self._current_adc_decim_ma()
            if v is not None:
                return v
        return self._current_adc_decim()

    def _current_weight_gross(self) -> Optional[float]:
        adc = self._choose_adc()
        if adc is None:
            return None
        return self.config.calibration.adc_to_gram(adc)

    def _current_weight_net(self) -> Optional[float]:
        gross = self._current_weight_gross()
        if gross is None:
            return None
        return gross - self._net_tare_g

    def _series_for_stability(self) -> List[float]:
        if self.config.use_one_euro and self.adc_window_decim_euro:
            return list(self.adc_window_decim_euro)
        if self.var_apply_ma.get() and self.adc_window_decim_ma:
            return list(self.adc_window_decim_ma)
        return list(self.adc_window_decim)

    def _is_stable(self) -> bool:
        series = self._series_for_stability()
        if len(series) < 5:
            return False
        try:
            std_th = float(self.ent_std_th.get())
        except Exception:
            std_th = self.config.stability_std_threshold_g
        if not self.config.calibration.is_ready() and not self.config.calibration.has_npoint():
            return False
        N = max(5, self._get_ma_window())
        tail = series[-N:]
        sigma_adc = statistics.pstdev(tail)
        adc_now = series[-1]
        gain = self.config.calibration.gain_at_adc(adc_now)
        if gain is None:
            return False
        g_sigma = abs(sigma_adc * gain)
        return g_sigma <= std_th

    # ===== イベント/保存 =====
    def _on_step_changed(self):
        try:
            self.config.display_step_g = float(self.cmb_step.get())
            self.config.save()
        except Exception:
            pass

    def _get_ma_window(self) -> int:
        try:
            return max(1, min(500, int(self.spn_ma.get())))
        except Exception:
            return max(1, self.config.ma_window)

    def _get_decim_factor(self) -> int:
        try:
            return max(1, min(128, int(self.spn_decim.get())))
        except Exception:
            return max(1, self.config.decim_factor)

    def _on_ma_changed(self):
        n = self._get_ma_window()
        self.config.ma_window = n
        self.config.save()

    def _on_decim_changed(self):
        d = self._get_decim_factor()
        self._decim_sum = 0.0
        self._decim_count = 0
        self.config.decim_factor = d
        self.config.save()

    def _on_apply_ma_changed(self):
        self.config.apply_ma_to_display = bool(self.var_apply_ma.get())
        self.config.save()

    def _on_toggle_euro(self):
        self.config.use_one_euro = bool(self.var_use_eu.get())
        self.config.save()

    def _apply_euro_params(self):
        try:
            minc = float(self.ent_eu_min.get())
            beta = float(self.ent_eu_beta.get())
            dco = float(self.ent_eu_d.get())
            errk = float(self.ent_eu_errk.get())
            erre = float(self.ent_eu_err_eps.get())
            fcmx = float(self.ent_eu_fcmax.get())
        except Exception:
            messagebox.showwarning("入力エラー", "OneEuro パラメータを数値で入力してください")
            return

        self.config.one_euro_min_cutoff = max(1e-6, minc)
        self.config.one_euro_beta = beta
        self.config.one_euro_d_cutoff = max(1e-6, dco)
        self.config.one_euro_err_boost_k = errk
        self.config.one_euro_err_eps_g = max(0.0, erre)
        self.config.one_euro_fc_max = max(0.1, fcmx)
        self.config.save()

        self._euro = OneEuroFilter(
            self.config.one_euro_min_cutoff,
            self.config.one_euro_beta,
            self.config.one_euro_d_cutoff,
            self.config.one_euro_err_boost_k,
            self.config.one_euro_err_eps_g,
            self.config.one_euro_fc_max,
        )
        self._euro.reset()
        self._update_euro_gain()
        self.var_status.set("OneEuro パラメータを適用しました")

    def _update_cal_info_label(self):
        cal = self.config.calibration
        if cal.is_ready():
            base = f"ZeroADC={cal.zero_adc:.6f}, SpanADC={cal.span_adc:.6f}, SpanW={cal.span_weight_g:g} g"
            if cal.has_npoint():
                stats = cal.stats()
                if stats:
                    rmse, mx = stats
                    base += f" | 多点:{len(cal.points)}点, RMSE={rmse:.3f} g, MaxErr={mx:.3f} g"
            self.var_cal_info.set(base)
        else:
            if cal.has_npoint():
                stats = cal.stats()
                if stats:
                    rmse, mx = stats
                    self.var_cal_info.set(f"多点:{len(cal.points)}点, RMSE={rmse:.3f} g, MaxErr={mx:.3f} g")
                else:
                    self.var_cal_info.set(f"多点:{len(cal.points)}点（統計不可）")
            else:
                self.var_cal_info.set("未校正（ゼロ→スパンの順に実施）")
        self._caln_refresh_table()

    def _update_unit_weight_label(self):
        if self.config.unit_weight_g and self.config.unit_weight_g > 0:
            self.var_unit_weight.set(f"単重(g): {self.config.unit_weight_g:.6f}")
        else:
            self.var_unit_weight.set("単重(g): ---")

    def _update_euro_gain(self):
        cal = self.config.calibration
        adc_now = self._current_adc_decim() or self._current_adc_raw()
        g_per_adc = cal.gain_at_adc(adc_now)
        self._euro.set_gain_g_per_adc(g_per_adc)

    # ===== 多点CAL: ヘルパ =====
    def _caln_refresh_table(self):
        if not hasattr(self, "tree_caln"):
            return
        self.tree_caln.delete(*self.tree_caln.get_children())
        for adc, g in self.config.calibration._sorted_points():
            self.tree_caln.insert("", tk.END, values=(f"{adc:.6f}", f"{g:.6f}"))
        n = len(self.config.calibration.points)
        stats = self.config.calibration.stats()
        if stats:
            rmse, mx = stats
            self.var_caln_stats.set(f"点数: {n}, RMSE: {rmse:.3f} g, MaxErr: {mx:.3f} g")
        else:
            self.var_caln_stats.set(f"点数: {n}, RMSE: -, MaxErr: -")

    def _caln_add_current(self):
        adc = self._choose_adc()
        if adc is None:
            messagebox.showinfo("情報", "ADC値が取得できていません（接続/安定を確認）")
            return
        try:
            w = float(self.ent_caln_w.get())
        except Exception:
            messagebox.showwarning("入力エラー", "既知重量(g) を数値で入力してください")
            return
        self.config.calibration.points.append((float(adc), float(w)))
        self.config.calibration._invalidate()
        self.config.save()
        self._update_cal_info_label()
        self._update_euro_gain()
        self.var_status.set(f"多点CAL: 追加 (ADC={adc:.6f}, g={w:.6f})")

    def _caln_del_selected(self):
        sel = self.tree_caln.selection()
        if not sel:
            return
        dels = set((float(self.tree_caln.item(iid, "values")[0]),
                    float(self.tree_caln.item(iid, "values")[1])) for iid in sel)
        self.config.calibration.points = [p for p in self.config.calibration.points if p not in dels]
        self.config.calibration._invalidate()
        self.config.save()
        self._update_cal_info_label()
        self._update_euro_gain()
        self._caln_refresh_table()

    def _caln_clear(self):
        if messagebox.askyesno("確認", "多点キャリブレーション点をすべて削除しますか？"):
            self.config.calibration.points.clear()
            self.config.calibration._invalidate()
            self.config.save()
            self._update_cal_info_label()
            self._update_euro_gain()
            self._caln_refresh_table()

    def _caln_sort(self):
        self.config.calibration.points = self.config.calibration._sorted_points()
        self.config.calibration._invalidate()
        self.config.save()
        self._caln_refresh_table()

    # ===== 描画 =====
    @staticmethod
    def _draw_led(canvas: tk.Canvas, on: bool):
        canvas.delete("all")
        color = "#43d17a" if on else "#c0c0c0"
        canvas.create_oval(2, 2, 16, 16, fill=color, outline="")


if __name__ == "__main__":
    app = App()
    app.mainloop()
