////////////////////////////////////////////
// 低コストUECS PAS方式CO2濃度計測ノード
// LU_PASCO2measNode.ino
// 2025.10.14 T.Hoshi ver. 1.0-D
// 2025.10.20 I2C通信の不安定要因の除去　ver. 1.1-D
// 2025.10.29 センサの電源コントロール　ver. 1.2-D
// 2025.11.14 タイミング調整して分低動作確認 ver. 1-3
////////////////////////////////////////////
//               012345678901
#define verNO   "LU-PASCO2B13" // 12 char.
#define verDATE "20251114" // 8 char.
// このプログラムは、無償提供する代わりに、動作結果に関する
// 一切の保証は致しません。自己責任でお使いください。また、
// 改造も自由ですが、公刊等をされる場合は、「近畿大学　星　
// 岳彦が開発した○○〇プログラムを用いた」と引用して下さい
// ますと、今後の研究開発の励みになりますので、どうぞよろ
// しくお願いします。
// このスケッチはWZnet W5500-EVB-Picoで動作させることを
// 前提にしています。
// 機器ごとの設定
#define NO_1
// 適切なmacアドレスをmac0-mac5に設定してください。macアドレスとは。
// 90:A2:DA:xx:xx:xxのような6個の16進数です。これをばらして0xの後に
// 転記してください。
#ifdef NO_1
 #define mac0 0x00
 #define mac1 0x00
 #define mac2 0x00
 #define mac3 0x00
 #define mac4 0x00
 #define mac5 0x00
#endif
/*
-----------------------------------------------------------------------------
LU-PASCO2 software are distributed under the 2 clause BSD license.

Copyright 2025, Takehiko Hoshi All rights reserved.

Redistribution and use in source and binary forms, with or without 
modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
    this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice, 
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
-----------------------------------------------------------------------------
LU-PASCO2ソフトウェアは、2条項BSDライセンスの下に配布されます。

Copyright 2025, Takehiko Hoshi All rights reserved.

  ソースコード形式かバイナリ形式か、変更するかしないかを問わず、以下の条件を
満たす場合に限り、再頒布および使用が許可されます。

1.ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、および下記免責
  条項を含めること。
2.バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の資料に、上記の
  著作権表示、本条件一覧、および下記免責条項を含めること。

  本ソフトウェアは、著作権者およびコントリビューターによって「現状のまま」提供
されており、明示黙示を問わず、商業的な使用可能性、および特定の目的に対する適合
性に関する暗黙の保証も含め、またそれに限定されない、いかなる保証もありません。
著作権者もコントリビューターも、事由のいかんを問わず、 損害発生の原因いかんを
問わず、かつ責任の根拠が契約であるか厳格責任であるか（過失その他の）不法行為で
あるかを問わず、仮にそのような損害が発生する可能性を知らされていたとしても、
本ソフトウェアの使用によって発生した （代替品または代用サービスの調達、使用の
喪失、データの喪失、利益の喪失、業務の中断も含め、またそれに限定されない）直接
損害、間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害について、
一切責任を負わないものとします。
-----------------------------------------------------------------------------
*/
/*
本ソフトウェアは農研機構黒崎様開発のUECSミドルウェアUARDECS_PICOを使っています。
https://github.com/H-Kurosaki/UARDECS_PICO
(1)コンパイル環境の構築(Arduino IDE1.8.19で確認済み)
	(a)RP2040用ボードライブラリのインストール
	ArduinoIDEを起動しファイル→環境設定→追加のボードマネージャのURLに以下のURLを入力します
	https://github.com/WIZnet-ArduinoEthernet/arduino-pico/releases/download/global/package_rp2040-ethernet_index.json

	(b)ツール→ボード→ボードマネージャより
	"Raspberry Pi Pico/RP2040 Ethernet"を検索しインストールします。

	(c)ツール→ボード→Raspberry Pi RP2040 Boards
	から使用する適切な機種名を選んでください(Wiznet W5500-EVB-Picoなど)
	
	(d)https://github.com/UECS/MyEthernet2
	"MyEthernet2"をインストールします。zipファイルでダウンロードしたあと、解凍してlibrariesフォルダ内に置いてください。
	
	以上の処理(一度行えば次から不要です)で最低限このファイルがコンパイル可能な環境が整います。

	以下の操作はボードを接続する事に毎回実行してください。
	(e)PCのUSBポートにボードを接続し、ツール→ボードより機種名が合っていることを確認します。

	(f)コンパイル前にツール→シリアルポートから接続中のボードのシリアルポート番号を選んでください。
	(ボードの個体ごとに違う値になる可能性があります)

	(g)IDE画面左上の→マークをクリックするとコンパイルとプログラムの転送が始まります。

	(h)転送が終わると自動的にボードがリセットされプログラムが実行されます。

(2)Arduino用のソースコードをUARDECS_PICOに移植する場合の注意点
	Arduino用のUARDECSおよびUARDECS_MEGAとソースコードレベルで互換性があります。
	ヘッダファイル(ライブラリ)の変更のみで大抵はコンパイルをそのまま通ります。
	ヘッダファイルには最低限"#include <Uardecs_pico.h>"の記述のみ必要ですが、
	Ethernet3.hは必須なので意図的に記述しています。

	ただし、ハードウェアに依存する部分は変更が必要です。
	本ライブラリが動作対象として想定しているPICO互換機のW5500-EVB-Picoでは以下のピンが最初から専有されています。
	(I/O)	PIN_ID	Note
	I	GPIO16	Connected to MISO on W5500
	O	GPIO17	Connected to CSn on W5500
	O	GPIO18	Connected to SCLK on W5500
	O	GPIO19	Connected to MOSI on W5500
	O	GPIO20	Connected to RSTn on W5500
	I	GPIO21	Connected to INTn on W5500
	I	GPIO24	VBUS sense - high if VBUS is present, else low
	O	GPIO25	Connected to user LED
	I	GPIO29	Used in ADC mode (ADC3) to measure VSYS/3

	Arduinoで動いていたデバイスも新たにRaspberry PI PICOで動かす場合は相性のテストが不可欠です。
	例えばI2Cを使用する場合、Arduinoと異なりbegin時にピン番号の指定が必要など、細部に差があります。
	Raspberry PI PICOは処理速度やメモリの搭載量でArduinoを超えますが電圧が3.3Vであること、
	ピンに流せる電流が4mA以下であることに注意してください。
	
(2)EEPROMについて
	本ライブラリはArduinoに搭載されていたEEPROMの代用としてRaspberry Pi Picoのプログラムフラッシュの未使用領域4096byteを保存領域に使用します。
	また、SRAMを4096byte分キャッシュ領域として使用する(起動後に確保されるので注意)ので、この分だけ利用可能なメモリが減少します。

	EEPROMの耐久性は書き換え回数のみはArduinoと同等（100,000回)ですが、4096byte単位でしか書き換えができないため、
	一部書き換えただけでも4096byteの領域全体の耐久性を損耗します。
	これらは書き換え頻度が少ないIPアドレスの保存やその他設定データの保存には深刻な影響はないかもしれません。
	しかし、データのロギングなど一定時間間隔で書き込むような用途には著しく不利になります。
	(おそらく)byte単位で書き換えが可能なArduinoより遥かに少ない回数で限界に達します。
	プログラム中でEEPROMを高頻度に書き換える場合、別途不揮発性のデータが保存可能なデバイスを接続する設計にすることを推奨します。
*/
#include <Uardecs_pico.h>
#include <MyEthernet2.h>
#include <Wire.h>

// センサ I2Cアドレス
#define PASCO2_ADRS 0x28
// LCD I2Cアドレス
#define LCD_ADRS 0x3E
// 押しボタンのポートセット
#define PB 0
// I2C set
#define LCD_SDA 4
#define LCD_SCL 5
// 作業変数
uint8_t i, value, caribC;
signed long waitSec, noRst ; // 計測値が得られない秒数

boolean readyData;
uint8_t reg[17]; // for PAS CO2 sensor
uint16_t  co2, ap;
//LCD表示用バッファ他
char lcd_buf[20] ;

//////////////////////////////////
// UARDECS メイン設定
//////////////////////////////////
// UARDECS用IP初期化　ショートピン(D2)を抜くとセーフモード(192.168.1.7)
const byte U_InitPin = 2; // digital D3 pin for Safe mode
const byte U_InitPin_Sense=HIGH;
// UARDECSのUECSノード情報設定
const char U_name[] PROGMEM = verNO;
const char U_vender[] PROGMEM = "Hoshi-lab.";
const char U_uecsid[] PROGMEM = "011009000040";
const char U_footnote[] PROGMEM = "Product on UARDECS-PICO by <A HREF=\"http://hoshi-lab.info/\">Kindai, hoshi-lab.</A>";
char U_nodename[20] = verNO;
UECSOriginalAttribute U_orgAttribute;
//////////////////////////////////
// html page setting
//////////////////////////////////
//Define variables for Web interface
signed long W_CO2conc ; // CO2濃度(ppm)
signed long W_cycle ; // 計測サイクル(s)
signed long W_atmosP ; // 大気圧(hPa)
signed long W_refCO2 ; // 校正CO2濃度(ppm)
signed long W_mode ; // 計測モード 
// Define Strings
const char S_mode[] PROGMEM = "計測モード" ;
const char S_CO2conc[] PROGMEM = "CO2濃度" ;
const char S_cycle[] PROGMEM = "計測間隔" ;
const char S_atmosP[] PROGMEM = "大気圧" ;
const char S_refCO2[] PROGMEM = "CO2校正値" ;
// 注釈
const char D_mode[] PROGMEM = "校正要50秒" ;
// 選択文字列
const char S_meas[] PROGMEM = "計測" ;
const char S_setup[] PROGMEM = "設定" ;
const char S_carib[] PROGMEM = "校正" ;
const char S_reset[] PROGMEM = "初期化" ;
const char *SS_mode[4]= { S_meas, S_setup, S_carib, S_reset } ;
//Unit Strings
const char U_ppm[] PROGMEM = "ppm" ;
const char U_sec[] PROGMEM = "秒" ;
const char U_press[] PROGMEM = "hPa" ;
// Dummy Strings
const char EMPTY[] PROGMEM= "" ;
const char** DUMMY = NULL ;
const int U_HtmlLine = 5; //Total number of HTML table rows.
struct UECSUserHtml U_html[U_HtmlLine] = {
//{ 名前,      入出力形式,     単位,     詳細説明,選択肢文字列,選択肢数,値,           最小値,最大値,小数桁数}
  { S_mode,    UECSSELECTDATA,EMPTY,   D_mode,  SS_mode,    4,      &(W_mode),    0,       3,  0},
  { S_CO2conc, UECSSHOWDATA,  U_ppm,   EMPTY,   DUMMY,      0,      &(W_CO2conc), 1,   32000,  0},
  { S_cycle,   UECSINPUTDATA, U_sec,   EMPTY,   DUMMY,      0,      &(W_cycle),   5,    4095,  0},
  { S_atmosP,  UECSINPUTDATA, U_press, EMPTY,   DUMMY,      0,      &(W_atmosP),  750,  1150,  0},
  { S_refCO2,  UECSINPUTDATA, U_ppm,   EMPTY,   DUMMY,      0,      &(W_refCO2),  1,    32000, 0}
  } ;
//////////////////////////////////
// UserCCM setting
//////////////////////////////////
enum {
CCMID_cnd,
CCMID_CO2conc,
CCMID_dummy
};
const int U_MAX_CCM = CCMID_dummy;
UECSCCM U_ccmList[U_MAX_CCM];
const char ccmNamecnd[] PROGMEM = "機器状態";
const char ccmNameCO2conc[] PROGMEM = "CO2濃度";
const char ccmTypecnd[] PROGMEM = "cnd.mPC";
const char ccmTypeCO2conc[] PROGMEM = "CO2conc.mPC";
const char ccmUnitcnd[] PROGMEM = "";
const char ccmUnitCO2conc[] PROGMEM = "ppm";
//　UECS-CCMのUARDECSへの設定関数
void UserInit(){
  //Set ccm list
  //        sender,ID,            Description,    Type name,      Unit, Priority, Decimal Digit,CCM Level
  UECSsetCCM(true, CCMID_cnd,     ccmNamecnd,     ccmTypecnd,     ccmUnitcnd,    29, 0, A_1S_0) ;
  UECSsetCCM(true, CCMID_CO2conc, ccmNameCO2conc, ccmTypeCO2conc, ccmUnitCO2conc,29, 0, A_10S_0) ;
  //必ずシールドに貼ってあるアドレスに書き換えて使用して下さい
  U_orgAttribute.mac[0] = mac0;
  U_orgAttribute.mac[1] = mac1;
  U_orgAttribute.mac[2] = mac2;
  U_orgAttribute.mac[3] = mac3;
  U_orgAttribute.mac[4] = mac4;
  U_orgAttribute.mac[5] = mac5;
}

//////////////////////////////////////////////
//// Libuary for PAS CO2 sensor operation ////
//// 2025.9.8 By T.Hoshi                  ////
//////////////////////////////////////////////
uint8_t PASCO2read(uint8_t regNo) { // register read
  Wire.beginTransmission(PASCO2_ADRS);
  Wire.write(regNo);
  Wire.endTransmission(false);
  delay(2);
  if(Wire.requestFrom(PASCO2_ADRS, 1U, 1U) > 0)
    return Wire.read();
  else return 0x0 ;
}

boolean PASCO2write(uint8_t regNo, uint8_t data) { // register write
  Wire.beginTransmission(PASCO2_ADRS);
  Wire.write(regNo);
  Wire.write(data);
  if(Wire.endTransmission() != 0) return false;
  else return true;
}

void PASCO2getAllReg(uint8_t* r) { // read out contents in all of reg,
  uint8_t pos;
  for(pos=0; pos<17; pos++) {
    r[pos] = PASCO2read(pos);
    Serial.print(pos, HEX);
    Serial.print("[");
    Serial.print(r[pos], HEX);
    Serial.print("-");
    Serial.print(r[pos], BIN);
    Serial.print("] ");
  }
  Serial.println("");
}

uint8_t PASCO2getSensorStatus(boolean res) {
  // 0x80: Sensor Ready,  0x40: PWM_DIS is high level 
  // 0x20: temp out of range,  0x10: VDD 5V out of range
  // 0x08: communication error 
  uint8_t dat;
  dat =  PASCO2read(0x01) & 0xB8;
  if(res) PASCO2write(0x01, 0x01); // clear com. error only
  return dat;
}

boolean PASCO2dataReady() { // check to get a new meas value
  if((0x10 & PASCO2read(0x07)) == 0x10) return true;
  else return false;
}

void PASCO2startCycMeas(uint16_t ap, uint16_t cy) {
  // start process for continus measurement with turning off the ABOC(auto adj.)
  // ap: Air Pressure (hPa), cy: meas. cycle time (sec interval)
  Serial.println("[Measurement Mode]");
  PASCO2setAtmosPres(ap); // set air pres.
  PASCO2setInterval(cy); // set intarval
  PASCO2write(0x04, 0x22); // start contnuous measurement
  delay(1000);
  Serial.println("start to measure cyclic.");
}

uint16_t PASCO2getAtmosPres() { // get air pressure (hPa)
  uint16_t ap;
  ap = PASCO2read(0x0B) * 0x100;
  ap += PASCO2read(0x0C);
  return ap;
}

void PASCO2setAtmosPres(uint16_t ap) { // set air pressure (hPa)
  PASCO2write(0x0B, (uint8_t)(ap / 0x100)); // set airpressH
  PASCO2write(0x0C, (uint8_t)(ap & 0xFF)); // set airpressL
  Serial.print("set AirP: ");
  Serial.print(PASCO2getAtmosPres());
  Serial.println(" hPa");
}

uint16_t PASCO2getInterval() { // get meas. interval (sec)
  uint16_t intv;
  intv = PASCO2read(0x02) * 0x100;
  intv += PASCO2read(0x03);
  return intv;
}

void PASCO2setInterval(uint16_t intv) { // set meas. interval (sec)
  PASCO2write(0x02, (uint8_t)(intv / 0x100)); // set intervalH
  PASCO2write(0x03, (uint8_t)(intv & 0xFF)); // set intervalL
  Serial.print("set interval: ");
  Serial.print(PASCO2getInterval());
  Serial.println(" sec.");
}

uint16_t PASCO2getCO2conc() { // get measurment data from sensor
  uint16_t co2, x;
  for(x = 0; x < 5; x++) { // retry roop until 5 times
    co2 = (uint16_t)PASCO2read(0x05) * 0x0100; // get MSB CO2 conc.
    co2 += PASCO2read(0x06); // get LSB CO2 conc.
    if(((PASCO2getSensorStatus(true) & 0x08) == 0)&& // IF com. no err.
       (co2 < 32000)) break; // getting varid value
    Serial.println("retry get CO2.");
    delay(600);
  }
  if(co2 >= 32000) { // senser error restart
    PASCO2softReset();
    co2 = 9999;
  }
  Serial.print("CO2: ");
  Serial.print(co2);
  Serial.println(" ppm");
  return co2;
}

void PASCO2forcedCarib(uint16_t ref) { // adjust CO2 conc. of ref.
  int x;
  Serial.println("[Calibration Mode]");
  PASCO2write(0x10, 0xA3); // Reset sensor
  delay(1000);
  Serial.print("Wait for 10 seconds. [");
  for(x=0; x<3; x++) {
    delay(3333);
    Serial.print("*");
    dispCaribCup();
  }
  Serial.println("]");
  PASCO2write(0x10, 0xFC); // reset compensation value
  delay(1000);
  PASCO2write(0x10, 0xBC); // reset ABOC content value
  delay(1000);
  PASCO2write(0x0D, (uint8_t)(ref / 0x0100)); // set reference CO2 conc.
  PASCO2write(0x0E, (uint8_t)(ref & 0x00FF));
  PASCO2setAtmosPres((uint16_t)W_atmosP); // set air pres.
  PASCO2setInterval(10); // set intarval to 10 sec.
  Serial.print("*** start forced calibration to adjusting ");
  Serial.print(ref);
  Serial.println(" ppm.");
  PASCO2write(0x04, 0x2A); // start calibration
  dispCaribCup();
  Serial.print("Wait for 30 seconds. [");
  for(x=0; x<12; x++) {
    delay(2500);
    Serial.print("*");
    dispCaribCup();
  }
  Serial.println("]");
  do { // wait for end of calibration
    delay(1000);
  } while((PASCO2read(0x04) & 0x0C) != 0x04);
  Serial.println("*** save the calibratoion value and disable ABOC.");
  PASCO2write(0x10, 0xCF); // save the calib. value
  delay(1000);
  PASCO2write(0x04, 0x20); // ABOC off and idle mode
  delay(1000);
  Serial.println("*** end of calibratoion.");
  PASCO2softReset();
}

void PASCO2softReset() { // soft reset
  PASCO2write(0x10, 0xA3);
  Serial.println("*** soft reset the sensor");
  delay(1000);
  // start contnuous measurement
  PASCO2startCycMeas((uint16_t)W_atmosP, (uint16_t)W_cycle);
  waitSec = 0;
  readyData = false;
  W_mode = 0;
  dispCO2(0);
  delay(1000);
}
//////////////////////////////////////////////

/////////////////////////////////////////////
////  Libuary for 16x2 LCD char. display ////
/////////////////////////////////////////////
void LCD_write(byte t_data) { //LCDにデータを送る
  Wire.beginTransmission(LCD_ADRS);
  Wire.write(0x40);
  Wire.write(t_data);
  Wire.endTransmission();
  delay(1);
}

void LCD_string(char t_data[]) { // LCDに文字列を表示させる
  for(i = 0; i < 16; i++) {
    Wire.beginTransmission(LCD_ADRS);
    Wire.write(0x40) ;
    Wire.write(t_data[i]) ;
    Wire.endTransmission() ;
    delay(1) ;
  }
}

void LCD_command(byte t_command) { //LCDに命令を実行させる
  Wire.beginTransmission(LCD_ADRS);
  Wire.write(0x00);
  Wire.write(t_command);
  Wire.endTransmission();
  delay(10);
}

void LCD_init() { // LCD初期設定
  LCD_command(0x38); // 8bit2行表示
  LCD_command(0x39); // イントラクションテーブル1選択
  LCD_command(0x14); // 1/5バイアス・F2周波数
  LCD_command(0x73); // コントラスト4(16-0)
//LCD_command(0x52); // 5Vコントラスト
  LCD_command(0x56); // 3Vコントラスト
  LCD_command(0x6C); // フォロワー回路ON フォロワー強度4(7-0)
  LCD_command(0x38); // イントラクションテーブル非選択
  LCD_command(0x01); // 表示クリア
  LCD_command(0x0C); // 表示開始。カーソルオフ、カーソルポジションオフ
}

void LCD_dispL0() { // 次を0行目から表示
  LCD_command(0x80|0x00);
}

void LCD_dispL1() { // 次を1行目から表示
  LCD_command(0x80|0x40);
}

void LCD_clsBuf() { // 表示バッファクリア
  char x;
  for(x=0 ; x<16 ; x++) lcd_buf[x]=' ' ;
}
//////////////////////////////////////////////

// 計測値の表示
void dispCO2(uint16_t value) { // CO2濃度液晶表示
  // モード表示
  LCD_dispL0();  
  //              0123456789012345
  String dispS = "[MEAS]     s cy.";
  dispS.toCharArray(lcd_buf, 17);
  dtostrf((float)W_cycle,4,0,&lcd_buf[7]) ;
  lcd_buf[11] = 's';
  LCD_string(lcd_buf);
  // CO2濃度
  LCD_dispL1();  
  //       0123456789012345
  dispS = " CO2:       ppm ";
  dispS.toCharArray(lcd_buf, 17);
  if(value != 0)
    dtostrf((float)value,5,0,&lcd_buf[6]) ;
  else { // 'Ready' disp. in case value is 0
    lcd_buf[6] = 'R'; lcd_buf[7] = 'e'; lcd_buf[8] = 'a';
    lcd_buf[9] = 'd'; lcd_buf[10] = 'y';
  }
  lcd_buf[4] = ' ';
  LCD_string(lcd_buf);
}

// キャリブレーション表示
void dispCaribHd() { // キャリブレーション中の型枠表示
  // モード表示
  LCD_dispL0();  
  //              0123456789012345
  String dispS = "[CARIB]      ppm";
  dispS.toCharArray(lcd_buf, 17);
  dtostrf((float)W_refCO2,4,0,&lcd_buf[8]) ;
  lcd_buf[7] = 0x7E; // →表示
  LCD_string(lcd_buf);
  // CO2濃度
  LCD_dispL1();  
  //       0123456789012345
  dispS = "________________";
  dispS.toCharArray(lcd_buf, 17);
  LCD_string(lcd_buf);
  caribC = 1; // *表示カウンタをリセット
}

// キャリブレーション表示
void dispCaribCup() { // キャリブレーション中のカウンタ増加
  char ps;
  LCD_dispL1();  
  for(ps = 0; ps < 16; ps++) {
    if(ps < caribC) lcd_buf[ps] = '*';
    else lcd_buf[ps] = '_';
  }
  LCD_string(lcd_buf);
  caribC++;
}

// 起動メッセージをLCDに表示
void LCDtitle() {
  LCD_init();
  //LCDにタイトル表示
  LCD_dispL0();
  String dispS = "**";
  dispS.concat(verNO);
  dispS.concat("**");
  dispS.toCharArray(lcd_buf, 17);
  LCD_string(lcd_buf) ;
  // LCDにバージョン表示を表示
  LCD_dispL1();
  dispS = verDATE;
  dispS.concat(" T.Hoshi");
  dispS.toCharArray(lcd_buf, 17);
  LCD_string(lcd_buf) ;
}

// UECSの変数を初期化
void UECS_Winit() {
  // W_modeの値が範囲外
  if((W_mode < 0)||(W_mode > 2)) {
    W_CO2conc = 0;
    W_cycle = 10;
    W_atmosP = 1013;
    W_refCO2 = 400;
    W_mode = 0;; 
  }
  readyData = false;
  waitSec = noRst = 0; // 計測待ち秒数クリア
}
//---------------------------------------------------------
//起動直後に１回呼び出される関数。
//様々な初期化処理を記述できる。
//この関数内ではUECSsetup()関数を呼び出さなくてはならない。
//必要に応じて処理を記述してもかまわない。
//---------------------------------------------------------
void setup()
{
  Serial.begin(9600);
  // 押しボタンのポートセット
  pinMode(PB, INPUT_PULLUP);
  // I2C初期化とスタート画面のLCD表示
  Wire.setSDA(LCD_SDA);
  Wire.setSCL(LCD_SCL);
  // 400 kHzでないとLCDとのバス通信で読みこみ不良
  Wire.setClock(400000); // I2Cのクロック
  Wire.begin();
  delay(1000);
  Serial.println("*** I2C and Serial initialized");
  // Display Title for LCD
  //LCD初期化
  LCDtitle();
  // UARDECS起動
  UECS_Winit(); // 不揮発性変数に初期値セット
  UECSsetup();
  U_ccmList[CCMID_cnd].value = 0;
  U_ccmList[CCMID_CO2conc].value = 0;
  // PASCO2センサが起動したことを確認できるまで待つ
  PASCO2softReset();
  do {
    delay(1000);
  } while((PASCO2getSensorStatus(true) & 0x80) == 0x00);
  Serial.println("*** confirmation of working PASCO2 sensor.");
  // reset PASCO2
  dispCO2(0); // 計測モードのLCD表示
}

//---------------------------------------------------------
//Webページから入力が行われ各種値を設定し不揮発性メモリに値を保存後、
//ページの再描画を行う前に以下の関数が呼び出される。
//---------------------------------------------------------
void OnWebFormRecieved() {
  switch(W_mode)  {
    case 0: // 計測なら何もしない
      break;
    case 1: // 設定ならセンサをソフトリセットして計測再開
      // reset PASCO2
      PASCO2softReset();
      break;
    case 2: // 校正ならセンサをリファレンスの濃度で校正(50秒所要)
      dispCaribHd();
      // start contnuous measurement every 10 sec. for carib.
      PASCO2startCycMeas((uint16_t)W_atmosP, (uint16_t)10);
      delay(1000);
      readyData = false;
      PASCO2forcedCarib((uint16_t)W_refCO2); // forced calib.
      delay(1000);
      // start contnuous measurement
      PASCO2startCycMeas((uint16_t)W_atmosP, (uint16_t)W_cycle);
      delay(1000);
      waitSec = 0;
      readyData = false;
      W_mode = 0;
      dispCO2(0);
      break;
    case 3: // 初期化なら設定値をリセットしてセンサ再起動
      UECS_Winit();
      PASCO2softReset();
      delay(1000);
      waitSec = 0;
      readyData = false;
      W_mode = 0;
      dispCO2(0);
      break;
    }
}

//---------------------------------------------------------
//setup()実行後に呼び出されるメインループ
//この関数内ではUECSloop()関数を呼び出さなくてはならない。
//UserEveryLoop()に似ているがネットワーク関係の処理を行う前に呼び出される。
//必要に応じて処理を記述してもかまわない。
//呼び出される頻度が高いため,重い処理を記述しないこと。
//---------------------------------------------------------
void loop(){  
  UECSloop();  
}

//---------------------------------------------------------
//メインループ
//システムのタイマカウント，httpサーバーの処理，
//UDP16520番ポートと16529番ポートの通信文をチェックした後，呼び出さされる関数。
//呼び出される頻度が高いため，重い処理を記述しないこと。
//---------------------------------------------------------
void UserEveryLoop(){
}

//---------------------------------------------------------
//１分に１回呼び出される関数
//---------------------------------------------------------
void UserEveryMinute(){
}

//---------------------------------------------------------
//１秒に１回呼び出される関数
//---------------------------------------------------------
void UserEverySecond(){
  waitSec++ ; // 一定秒数計測値が得られなければCO2センサをリスタート
  if(waitSec > (W_cycle + (W_cycle / 10))) {
    waitSec = 0;
    noRst++ ;
    PASCO2softReset();
  }

  if(readyData == true) { 
     Serial.println("*** get data!"); 
    // 計測データの準備ができた次のステップで読み込み
    W_CO2conc = (signed long)PASCO2getCO2conc();
    waitSec = 0;
    readyData = false;
    dispCO2((uint16_t)W_CO2conc);
    U_ccmList[CCMID_CO2conc].value = W_CO2conc;
    U_ccmList[CCMID_cnd].value = (PASCO2getSensorStatus(true) & 0x30);
  }
  // 計測データが用意できたかチェック
  if(PASCO2dataReady()) {
    readyData = true;
    Serial.println("*** data ready!");
    dispCO2(0);
  }
  // 押しボタンが押されたらセンサレジスタ表示
  if(digitalRead(PB) == 0) {
    uint8_t rd[17], id;
    PASCO2getAllReg(rd);
    LCD_dispL0();
    for(id = 0; id < 16; id+=2) {
      if((rd[id / 2] / 0x10) >= 0x0A) lcd_buf[id] = (rd[id / 2] / 0x10) + 0x37;
      else lcd_buf[id] = (rd[id / 2] / 0x10) + 0x30;
      if((rd[id / 2] & 0x0F) >= 0x0A) lcd_buf[id + 1] = (rd[id / 2] & 0x0F) + 0x37;
      else lcd_buf[id + 1] = (rd[id / 2] & 0x0F) + 0x30;
    }
    LCD_string(lcd_buf) ;
    LCD_dispL1();
    for(id = 0; id < 16; id+=2) {
      if((rd[id / 2 + 8] / 0x10) >= 0x0A) lcd_buf[id] = (rd[id / 2 + 8] / 0x10) + 0x37;
      else lcd_buf[id] = (rd[id / 2 + 8] / 0x10) + 0x30;
      if((rd[id / 2 + 8] & 0x0F) >= 0x0A) lcd_buf[id + 1] = (rd[id / 2 + 8] & 0x0F) + 0x37;
      else lcd_buf[id + 1] = (rd[id / 2 + 8] & 0x0F) + 0x30;
    }
    LCD_string(lcd_buf) ;
  }
  Serial.print("### 1sec. loop end. (waitSec=");
  Serial.print(waitSec);
  Serial.print(", noRst=");
  Serial.print(noRst);
  Serial.println(")");
}
