from machine import Pin, I2C, UART
import rp2
import network
import ambient
import ssd1306
import e220
import time
import ujson
import gc

# 初期化処理
uart = UART(0, baudrate=9600, tx=Pin(12), rx=Pin(13))
i2c = I2C(0, scl=Pin(5), sda=Pin(4))
disp = ssd1306.SSD1306_I2C(128, 64, i2c)

# コンフィグの読み込み
# （JSON形式で各種設定値を保存しておく）
with open("config.json", 'r') as f:
    cfg = ujson.load(f)

# Lora
m0 = Pin(6, Pin.OUT)
m1 = Pin(7, Pin.OUT)
aux = Pin(14, Pin.IN)
lora = e220.E220(uart, m0, m1, aux)

# Wi-Fi
rp2.country('JP')
wlan = network.WLAN(network.STA_IF)
wlan.active(True)
wlan.connect(cfg["SSID"], cfg["PASS"])

while not wlan.isconnected():
    print("connecting wifi ...")
    time.sleep(0.5)

lora.initialize_config(cfg["Lora_address"], cfg["Lora_channel"])

disp.fill(0)
disp.text("---  WiFi  ---", 5, 5, 1)
disp.text(f"{wlan.ifconfig()[0]}", 5, 15, 1)
disp.text("---  Lora  ---", 5, 25, 1)
disp.text(f"ADDR: {lora.addr}  CH: {lora.ch}", 5,  35, 1)

disp.show()

t0 = time.ticks_ms()
tick = 0
dispx = 5
disp_flags = [0, 0]
while True:
    try:
        rssi, d = lora.read()
        if rssi:
            # data
            # d1: Temp_flow
            # d2: RH_flow
            # d3: Co2
            # d4: Temp_box
            # d5: RH_box
            # d6: Soil moist
            # d7: Bat
            # d1: Stat0
            # d2: Stat1
            # d3: flag0
            # d4: RSSI
            rx_addr = d[0]
            rx_ch   = d[1]
            env_data = {
                "d1": d[2],
                "d2": d[3],
                "d3": (d[4] << 8) + d[5],
                "d4": d[6],
                "d5": d[7],
                "d6": (d[8] << 8) + d[9],
                "d7": (d[10] << 8) + d[11],
            }
            stat_data = {
                "d1": d[12],
                "d2": d[13],
                "d3": d[14],
                "d4": rssi,
            }

            # クラウドへ送信
            gc.collect()    # メモリ確保
            am = ambient.Ambient(cfg["Ambient_id1"], cfg["Ambient_key1"])
            r = am.send(env_data)
            am = ambient.Ambient(cfg["Ambient_id3"], cfg["Ambient_key3"])
            r = am.send(stat_data)

            # ディスプレイ表示
            disp_flags[rx_addr - 1] = 1
    except OSError:
        gc.collect()

    # ディスプレイ表示処理
    tick += 1
    if tick >= 20:
        tick = 0
        for i in range(2):
            if disp_flags[i]:
                disp.text("|", dispx, 45 + i * 10, 1)
            else:
                disp.text(".", dispx, 45 + i * 10, 1)
            disp_flags[i] = 0
        disp.show()
        dispx += 5
        if dispx >= 120:
            disp.fill_rect(5, 45, 130, 65, 0)
            dispx = 5
    time.sleep(0.5)
