/*****  ATtiny1604を使った周波数カウンタ  *****/
//      計測値は液晶表示とシリアル出力
//      "tiny1614_fcnt_tcb04.ino"  2025-12-17
//  PA1 11pin シリアル出力データ 9600BPS
//  PA2 12pin キャプチャタイミング RTCによる4Hz出力
//  PA3 13pin 測定クロック入力 (pull up)
//  PA4  2pin キャプチャ割り込み確認
//  PA5  3pin out
//  PA6  4pin in  信号レベル入力
//  PA7  5pin in  電池電圧低下チェック入力(1.1V)
//  PB0  9pin I2C液晶 SCL
//  PB1  8pin I2C液晶 SDA
//  PB2(7),PB3(6)  32.768kHz時計用水晶を接続
//  
//  クロックが無いとキャプチャデータが上がってこない
//  PITを使って1秒間をチェックしてキャプチャがないと0Hzと表示
//  CYC_HZで測定できる最低最高周波数が決まる
//  液晶表示器 AQM1602Yだと8つのCG_RAMが使えるが
//             AQM0802Aだと6つになってしまう CG_RAMデータを変更
#include "i2c_lcd_aqm1602_tn.h"   // LCD制御ヘッダーファイル
                                  // ★wire.hは使わない
i2c_lcd_aqm1602_tn LCD(0x3E);     // LCD I2Cアドレスを設定
//  タイトル
const char  pgm_ttl[]  = "ATtiny1614(LCD) Frq Cnt (2025-12-17)\r\n";
const char  pgm_ttl1[] =  {"FCnt1614"};
const char  pgm_ttl2[] =  {"25-12-17"};
const char *pgm_tlcd[] =  {pgm_ttl1, pgm_ttl2};
/*****  マクロ    *****/
#define DIMSIZ(a)   (sizeof(a)/sizeof(*a))  // 配列のﾃﾞｰﾀ数を返す
/*****  文字出力バッファ   *****/
char str_bff[64];   // 64文字 終端のnull含めて
                    // longは10桁 
/*****  シリアル出力    *****/
void tx(char c)
{
    while(!(USART0.STATUS & (1 << 5)));   // DREIF 送信できるまで待つ
    USART0.TXDATAL = c;     // 1文字出力
}
/*****  シリアル文字列出力    *****/
void txstr(const char *s)
{
    while(*s != '\0'){    // Nullまで
      tx(*s);             // 1文字出力
      s++;                // 次文字
    }
}
/*****  書式付シリアル出力     *****/
void txprintf(const char *s, ...)
{
va_list vp;
    va_start(vp,s);
    vsnprintf(str_bff, sizeof str_bff, s, vp);
    txstr(str_bff);
    va_end(vp);
}
/*****  RTC,PIT busy待ち *****/
//  両方とも0になるのを待つ
void rtcbusy(void) {
  while((RTC.STATUS & 0x0F) ||  // RTC busy ?
        (RTC.PITSTATUS & 1));   // PIT busy ?
}
/*****  16Hz PIT タイマーデータ    *****/
volatile byte tm_16hz;        // 16Hzダウンカウントタイマー
volatile byte tm16_lowbat;    // Low Batチェックタイマー
                              // 16Hzでダウンカウント
volatile byte f_1sec;         // 1秒フラグ
volatile byte cnt16_capchk;   // キャプチャ有無チェックカウンタ
                              // 16Hzでupカウント
                              // キャプチャでゼロクリア
/*****  16Hz PIT タイマー処理    *****/
//  loop内で毎回実行 16Hz=62.5ms
void pit16hz(void)
{
static byte cnt_16;           // 1秒チェックカウンタ
  if(RTC.PITINTFLAGS & 1){    // PIT 16Hz?
    RTC.PITINTFLAGS = 1;      // PITフラグクリア
//  ダウンカウントタイマ
    if(tm_16hz)       tm_16hz--;      // 16Hzタイマー   
    if(tm16_lowbat)   tm16_lowbat--;  // LowBatチェックタイマー
//  1秒チェック
    cnt_16++;                 // 16回カウント
    if(cnt_16 >= 10){         // 1秒経過
      cnt_16 = 0;
      f_1sec = 1;             // １秒フラグをオン
    }
//  upカウントチェック
    if(cnt16_capchk < 255)  cnt16_capchk++; // max255でカウントアップ
  }
}
/*************************/
/*      キャラジェネ      */
/*************************/
#define LCD_CGRAM   6     // 液晶のCG RAM エリアの大きさ 8 or 6
                          // AQM0802だと6つに減少
#if LCD_CGRAM == 8        // AQM1602Yタイプは8
#define CG_BAR0     0x00    // 0 _
#define CG_BAR1     0x01    // 1 |
#define CG_BAR2     0x02    // 2 ||
#define CG_BAR3     0x03    // 3 |||
#define CG_BAR4     0x04    // 4 ||||
#define CG_BAR5     0x05    // 5 |||||
#define CG_BATM     0x06    // 6 Bat M
#define CG_BATL     0x07    // 7 Bat L
/*****  キャラジェネデータ   *****/
//  コード0x00～0x07
const uint8_t cg0[] = {    // BAR0 _
    0b00000,    // 0
    0b00000,    // 1
    0b00000,    // 2
    0b00000,    // 3
    0b00000,    // 4
    0b00000,    // 5
    0b00000,    // 6
    0b11111,    // 7
};
const uint8_t cg1[] = {    // BAR1 |
    0b10000,    // 0
    0b10000,    // 1
    0b10000,    // 2
    0b10000,    // 3
    0b10000,    // 4
    0b10000,    // 5
    0b10000,    // 6
    0b11111,    // 7
};
const uint8_t cg2[] = {    // BAR2 ||
    0b11000,    // 0
    0b11000,    // 1
    0b11000,    // 2
    0b11000,    // 3
    0b11000,    // 4
    0b11000,    // 5
    0b11000,    // 6
    0b11111,    // 7
};
const uint8_t cg3[] = {    // BAR3 |||
    0b11100,    // 0
    0b11100,    // 1
    0b11100,    // 2
    0b11100,    // 3
    0b11100,    // 4
    0b11100,    // 5
    0b11100,    // 6
    0b11111,    // 7
};
const uint8_t cg4[] = {    // BAR4 ||||
    0b11110,    // 0
    0b11110,    // 1
    0b11110,    // 2
    0b11110,    // 3
    0b11110,    // 4
    0b11110,    // 5
    0b11110,    // 6
    0b11111,    // 7
};
const uint8_t cg5[] = {    // BAR5 |||||
    0b11111,    // 0
    0b11111,    // 1
    0b11111,    // 2
    0b11111,    // 3
    0b11111,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b11111,    // 7
};
const uint8_t cg6[] = {  // BAT Midle
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b11111,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b11111,    // 7 ← 8行目にもデータを入れられる
};
const uint8_t cg7[] = {  // BAT L
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b10001,    // 4
    0b10001,    // 5
    0b10001,    // 6
    0b11111,    // 7
};
const uint8_t *cg_tbl[]={
    cg0, cg1, cg2, cg3, cg4, cg5, cg6, cg7, // 8データ
};
#endif
#if LCD_CGRAM == 6        // AQM0802Yタイプ
#define CG_BAR0     0x5F    //   _
#define CG_BAR1     0x0B    //   |
#define CG_BAR2     0x00    // 0 ||
#define CG_BAR3     0x01    // 1 |||
#define CG_BAR4     0x02    // 2 ||||
#define CG_BAR5     0x03    // 3 |||||
#define CG_BATM     0x04    // 4 Bat M
#define CG_BATL     0x05    // 5 Bat L
/*****  キャラジェネデータ   *****/
//  コード0x00～0x05
const uint8_t cg2[] = {    // BAR2 ||
    0b11000,    // 0
    0b11000,    // 1
    0b11000,    // 2
    0b11000,    // 3
    0b11000,    // 4
    0b11000,    // 5
    0b11111,    // 6
    0b00000,    // 7
};
const uint8_t cg3[] = {    // BAR3 |||
    0b11100,    // 0
    0b11100,    // 1
    0b11100,    // 2
    0b11100,    // 3
    0b11100,    // 4
    0b11100,    // 5
    0b11111,    // 6
    0b00000,    // 7
};
const uint8_t cg4[] = {    // BAR4 ||||
    0b11110,    // 0
    0b11110,    // 1
    0b11110,    // 2
    0b11110,    // 3
    0b11110,    // 4
    0b11110,    // 5
    0b11111,    // 6
    0b00000,    // 7
};
const uint8_t cg5[] = {    // BAR5 |||||
    0b11111,    // 0
    0b11111,    // 1
    0b11111,    // 2
    0b11111,    // 3
    0b11111,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b00000,    // 7
};
const uint8_t cg6[] = {  // BAT Midle
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b11111,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b11111,    // 7 ← 8行目にもデータを入れられる
};
const uint8_t cg7[] = {  // BAT L
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b10001,    // 4
    0b10001,    // 5
    0b10001,    // 6
    0b11111,    // 7
};
const uint8_t *cg_tbl[]={
    cg2, cg3, cg4, cg5, cg6, cg7, // 6データ
};
#endif
/*****  バー表示データ  *****/
const char cg_bardata[] = {
    CG_BAR0, CG_BAR1, CG_BAR2, CG_BAR3, CG_BAR4, CG_BAR5,
};
/*****  キャラジェネ設定    *****/
//  液晶CG RAMデータを設定
void lcdcgset(void)
{
byte i;
    for(i = 0; i < DIMSIZ(cg_tbl); i++){          // データ数
        LCD.createChar(i, (uint8_t *)cg_tbl[i]);  // キャラジェネ設定  
    }
}
/*****  LowBat表示        *****/
//  LowBatならCG_BATL,CG_BATMを右上端に点滅
//  okならspaceに
//  1秒サイクルでチェック
void lowbat(void)
{
char c;
static byte blk = 0;          // LowBat電池点滅フラグ
  if(tm16_lowbat == 0){       // タイムアップ
    blk ^= 1;                 // 点滅フラグ反転
    tm16_lowbat = 16;         // タイマー1秒セット
    if(AC0.STATUS & (1 << 4))   c = ' ';  // 電池電圧ok
    else{                                 // Low BAT検出
      if(blk == 0)        c = CG_BATL;    // Lマーク
      else                c = CG_BATM;    // Mマーク
    }
    LCD.setCursor(7, 1);      // LCD 下段右端
    LCD.write(c);             // 上右端に電池マーク
  }
}
/*****  信号レベル表示    *****/
//  ADC0 AIN6=PA6でA/D変換
//  10bitを32回累積で0～32736の数値が出てくる
//  4Hz(0.25秒)サイクルで変換
//  25本のバーで表示 1文字あたり5本で5文字
//  変換完了フラグADC0.INTFLAGSはADC0.RES読み出しでクリア
void siglvl(void)
{
word d;
byte i, m;
char c;
  if(ADC0.INTFLAGS & 1){    // A/D変換完了
    d = ADC0.RES;           // 変換結果 (1023*32):0～32736
    if(d >= 32736)    m = 25;       // 25本が最大
    else              m = d / 1364; // 0～32736を0～24本に
    LCD.setCursor(2, 1);    // LCD下段2文字目から5文字
    for(i=0; i < 5; i++){   // 1文字あたり5本のバー 5文字loop
      if(m == 0){           // 0本 バーなし
        c = CG_BAR0;        // 「_」で
      }
      else if(m < 5){       // 1～5本
        c = cg_bardata[m];  // バー1～5
        m = 0;
      }
      else{                 // 6本以上
        c = CG_BAR5;        // バー5本
        m -= 5;
      }
      LCD.write(c);         // 液晶1文字書き込み
    }                       // 5文字loop
  }  
}
/*****  カウントデータ  *****/
#define  CYC_HZ   4   // キャプチャ繰り返し周波数(Hz)
                      // 毎秒あたりのキャプチャ回数 (2^n)
                      // 測定可能最低周波数
                      // 65536を乗ずると測定可能最大周波数
volatile uint32_t cnt_add;    // キャプチャ加算値(32bit)
volatile uint32_t cnt_wrd;    // 書き込みデータ(チェック用)
volatile byte f_capok;        // キャプチャ完了フラグ
volatile byte f_cap1st;       // 初めてのキャプチャフラグ
volatile uint32_t cap_bff[CYC_HZ];  // キャプチャデータ書き込みバッファ
volatile byte     cap_cnt;          // キャプチャデータ数
volatile byte     cap_wrp;          // キャプチャデータ書き込みポインタ
/*****  キャプチャデータ書き込み処理    *****/
//  d:キャプチャデータ
void capbffwr(uint32_t d)
{
  cnt_wrd = d;            // 書き込みデータを保存
  cap_bff[cap_wrp] = d;   // データ書き込み
  cap_wrp++;              // データ書き込みポインタ+1
  if(cap_wrp >= CYC_HZ){  // バッファいっぱい
    cap_wrp = 0;          // 先頭に
  }
  if(cap_cnt < CYC_HZ){   // バッファに置いたデータ数
    cap_cnt++;            // データ数+1 maxがCYC_HZ
  }
}
/*****  キャプチャデータクリア    *****/
void capbffclr(void)
{
  cli();                  // 割り込み禁止で
  cap_wrp = 0;            // 書き込み位置先頭に
  cap_cnt = 0;            // データ数クリア
  sei();                  // 割り込み再開
}
/*****  周波数計算     *****/
//  キャプチャデータから周波数を計算
uint32_t frqcapbff(void)
{
uint32_t d;   // 加算値
byte n, i;
  d = 0;                  // 加算値クリア
  cli();                  // 割り込み禁止で
  n = cap_cnt;            // キャプチャ書き込み回数
  if(n != 0){             // 回数0ならスキップ
    for(i = 0; i < n; i++){   // 書き込み回数
      d += cap_bff[i];        // バッファのデータを加算
    }                         // 最大は4Hz分
  }
  sei();                    // 割り込み再開
  if(n != 0){               // 1回以上の時
//  txprintf("%ld/%d:", d, n); // (!!!)
    d = (d * CYC_HZ) / n;      // 1Hzでの周波数に 
//  txprintf("%ld\r\n", d);    // (!!!)
  }
  return d;                 // 回数0なら0でリターン
}
/*****  TCB0 キャプチャ割り込み   *****/
//  キャプチャデータ読み出しで割り込みフラグクリア
//  PA4でタイミングをチェック
ISR(TCB0_INT_vect) 
{
static volatile uint16_t d, d1, d2;    // 16bit カウントデータ
  PORTA.OUTSET  = (1 << 4);   // PA4 H 2pin
  d = TCB0.CCMP;              // 16bitキャプチャデータ
  if(f_cap1st == 0){          // はじめての処理
    d1 = d;                   // 16bitキャプチャー値
    cnt_add  = 0;             // カウント加算値(32bit)ゼロから
    f_cap1st = 1;
  }
  else{                       // 2回目以降
    d2 = d;                   // 16bitキャプチャー値
    d  = d2 - d1;             // カウントの差を (16bit)
    cnt_add += (uint32_t)d;   // 32bit値で総数に加算 
    capbffwr(cnt_add);        // キャプチャデータバッファに書き込み
    d1 = d2;                  // 次カウント用に残す
    cnt_add = 0;              // カウント加算値(32bit)ゼロに
    f_capok = 1;              // キャプチャok
  }
  PORTA.OUTCLR  = (1 << 4);   // PA4 L
}
/*****  ＳＥＴＵＰ   *****/
void setup() {
  cli();                    // 割込禁止
  PORTA.OUT   = 0b00000010;
  PORTA.DIR   = 0b00110110;
  //              |||||||+---- PA0 10pin UPDI
  //              ||||||+----- PA1 11pin TXD
  //              |||||+------ PA2 12pin EVOUT0 RTC OVF
  //              ||||+------- PA3 13pin (in) 測定クロック入力 ←
  //              |||+-------- PA4  2pin キャプチャ割り込みのチェック
  //              ||+--------- PA5  3pin (out)
  //              |+---------- PA6  4pin (in) 信号レベル入力
  //              +----------- PA7  5pin (in) 電池電圧チェック
  PORTB.DIR   = 0b00000000;
  //                  |||+---- PB0  9pin I2C LCD SCL
  //                  ||+----- PB1  8pin I2C LCD SDA
  //                  |+------ PB2  7pin TOSC2 32.768kHz
  //                  +------- PB3  6pin TOSC1 XTAL
//  ポート機能 pull up,デジタル入力オフ
  PORTA.PIN3CTRL = 0b00001000;  // PA3 13pin クロック入力 pull up
  //                     +----- pull up有り
  PORTA.PIN6CTRL = 0b00000100;  // PA6 4pin 信号レベル入力 デジタル禁止
  PORTA.PIN7CTRL = 0b00000100;  // PA7 5pin 電池電圧入力   デジタル禁止
  //                 |   |+++-- デジタル入力無効
  //                 |   +----- pullupなし
  //                 +--------- 反転I/Oなし
//  代替ポート
  PORTMUX.CTRLA  = 0b00000001;   // 代替ポート
  //                   || ||+--- EVOUT0  PA2 EVOUT0 イベント出力
  //                   || |+---- EVOUT1  PB2
  //                   || +----- EVOUT2  PC2
  //                   |+------- LUT0out PB4
  //                   +-------- LUT1out PC1
  PORTMUX.CTRLB  = 0b00000001;  // 代替ポート
  //                      | +-- USART0 TXをPA1に
  //                      +---- SPIO
//  32.768kHz XTAL
//  _PROTECTED_WRITE(CLKCTRL.XOSC32KCTRLA, 0x00); // XTALオフ
//  while(CLKCTRL.MCLKSTATUS & (1 << 6));  // XOSC32KSの0を待つ
//        リセット起動なら↑の手順は不要  いきなりオン↓でOK
  _PROTECTED_WRITE(CLKCTRL.XOSC32KCTRLA, 0b00000001); // 発振イネーブル
  //                                         || ||+-- ENABLE
  //                                         || |+--- RUNSTBY
  //                                         || +---- SEL
  //                                         ++------ CSUT 1kサイクル
  //      CSUT 01で16kサイクル、10で32k、11で64kサイクルの起動時間待ち
//  RTC オーバーフローでTCB0キャプチャ
  rtcbusy();                  // RTC,PIT ビジー待ち
  RTC.CLKSEL  = 0b00000010;
  //                    ++---- XOSC32K 32.768kHzHz水晶
  RTC.CTRLA   = 0b00110001;
  //              |||||  +---- RTCEN RTC周辺機能許可
  //              |++++------- CLK_RTC 1/64 →512Hz
  //              +----------- RUNSTBY
  RTC.PER     = (512 / CYC_HZ) - 1;    // 512/128 →4Hz
  RTC.PITCTRLA  = 0b01010001;  
  //                 ||||  +-- PITEN PIT有効
  //                 ++++----- PERIOD 32.768kHz/2048 →16Hz
//  タイマーA
  TCA0.SPLIT.CTRLA     = 0b00000000;
  //                            |||+-- ENABLE off
  //                            +++--- CLKSEL
  TCA0.SINGLE.CTRLD    = 0;           // TCA0を16bitモードに
  TCA0.SINGLE.EVCTRL   = 0b00000001;
  //                           |||+-----  CNTEI イベントカウント
  //                           +++------  EVACT ↑エッジ CLK_TCAへ
  TCA0.SINGLE.CTRLA    = 0b00000001;
  //                           |||+-----  ENABLE 動作開始
  //                           +++------  CLKSEL プリスケーラ1/1
//  タイマーB
//  RTCのオーバーフローでキャプチャ
//  カウントクロックはCLK_TCA
  TCB0.CTRLA    = 0b00000101;
  //                 | | ||+-- ENABLE
  //                 | | ++--- CLKTCA TCAからのクロック
  //                 | +------ SYNCUPD
  //                 +-------- RUNSTBY
  TCB0.CTRLB    = 0b00000010;
  //                 ||| +++-- CNTMODE キャプチャモード
  //                 ||+------ CCMPEN
  //                 |+------- CCMPINIT
  //                 +-------- ASYNC
  TCB0.EVCTRL   = 0b00000001;   // イベント
  //                 | |   +-- CAPTEI キャプチャ有効
  //                 | +------ EDGE
  TCB0.INTCTRL  = 0b00000001;   // 割り込み
  //                       +-- CAPT キャプチャ割り込みon
//  イベント制御 TCAクロック入力
  EVSYS.SYNCCH0     = 0x10;  // ch0入力:PA3
  EVSYS.SYNCUSER0   = 0x01;  // ch0出力:TCA0 クロック
//  イベント制御 TCBキャプチャ
//  RTCのオーバーフローでキャプチャ
//  PA2にタイミングを出力
  EVSYS.ASYNCCH1    = 0x08;  // ch1入力:RTC_OVF
  EVSYS.ASYNCUSER0  = 0x04;  // ch1出力1:TCB0イベント (キャプチャ)
  EVSYS.ASYNCUSER8  = 0x04;  // ch1出力2:EVOUT0(PA2) 出力 
//  イベント制御 A/D変換開始
  EVSYS.ASYNCCH3    = 0x0A;  // ch3入力:PIT 1/8192(4Hz)
  EVSYS.ASYNCUSER1  = 0x06;  // ch3出力:ADC0へ
//  シリアル
// 9600BPSでシリアル出力  受信はなし
  USART0.CTRLB = 0b01000000;
  //               || ||||+-- MPCE
  //               || ||++--- RXMODE
  //               || |+----- ODME
  //               || +------ SFDEN
  //               |+-------- TXEN 送信有効
  //               +--------- RXEN 受信はしない
  USART0.CTRLC = 0b00000011;
  //               |||||+++-- CHSIZE 8BIT
  //               ||||+----- SBMODE 1stop
  //               ||++------ PMODE  Parity無し
  //               ++-------- CMODE  非同期USART
  USART0.BAUD  = (4L * F_CPU) / 9600L;  // 20MHzなら8333
  sei();                  // 割込有効
//  アナログコンパレータ
  VREF.CTRLA   = 0b00010001;  // VREF
  //                ||| +++-- DAC0REFSEL AC0比較電圧 1.1V
  //                +++------ ADC0REFSEL ADC0 Vref   1.1V
  AC0.CTRLA    = 0b00000011;  // アナログコンパレータ0
  //               |||| ||+-- AC0イネーブル
  //               |||| ++--- ヒステリシス 10mV
  //               ||++------ エッジ↑↓
  //               |+-------- OUTEN 出力しない
  //               +--------- RUNSTBY sleepで停止
  AC0.MUXCTRLA = 0b00000010;  
  //               |  || ++-- 反転入力 VREFに
  //               |  ++----- 非反転入力 AINP0:PA7
  //               +--------- INVERT しない
//  A/D変換  ADC6:AIN6:PA6 4pin
  ADC0.CTRLA  = 0b00000001;
  //              |    | +--- ENABLE  許可
  //              |    +----- RESEL 10bit
  //              +---------- RUNSTDBY
  ADC0.CTRLB  = 0b00000101;   // サンプル回数
  //                   +++--- 32回 15bit (max 32736)
  ADC0.CTRLC  = 0b01000100;
  //               ||| +++--- PRESC  10MHz/32=312.5kHz        
  //               |++------- REFSEL 内部Vref 1.1V (VREF.CTRLA)
  //               +--------- SMAPCAP Hiレベル
  ADC0.MUXPOS = 0b00000110;   // MPX
  //                 +++++--- AIN6:PA6 4pin 
  ADC0.INTCTRL= 0b00000000;    // 割り込み
  //                    |+--- RESRDY 変換完了割込なし
  //                    +---- WCMP
  ADC0.EVCTRL = 1;            // イベント入力(PIT)でA/D変換開始指令
//  I2C液晶 ★wire.hは使わない
  LCD.begin(16, 2, 100, LCD_V3R3);  // 液晶初期化 16文字x2行,100kHz,3.3V
                                    // (5V:LCD_V5R0,3.3VならLCD_V3R3に)
  LCD.setContrast(30);              // コントラスト設定
  lcdcgset();       // 液晶キャラジェネ設定 (8つの文字)
}
/*****  ＬＯＯＰ  *****/
//  1秒周期のキャプチャを待ってシリアル出力
//    1234.567kHz
//  1.5秒以上パルスが来ないと0.000kHzを出力
void loop() { 
static uint32_t d;      // 周波数データ
static byte f_disp;     // 周波数表示(出力)フラグ
static byte loop_exc;   // loop実行処理区分
  while(!(CLKCTRL.MCLKSTATUS & (1 << 6)));  // XOSC32KSの1を待つ
  while(1){
    pit16hz();        // 16Hz(62.5ms)PIT タイマー処理
    switch(loop_exc){
     case 0:   // 初めて タイトル表示
      txstr(pgm_ttl);                       // タイトル出力
      txprintf("F_CPU:%luHz\r\n", F_CPU);   // 動作周波数
      LCD.setCursor(0, 0);      // LCD上段
      LCD.print(pgm_tlcd[0]);   // タイトル上段
      LCD.setCursor(0, 1);      // LCD左下
      LCD.print(pgm_tlcd[1]);   // タイトル下段
      tm_16hz = 32;             // 2秒待ち
      loop_exc++;
      break;
    case 1:   // 2秒待ちして計測開始
      if(tm_16hz == 0){         // 2秒経過
        LCD.clear();
        LCD.setCursor(6, 0);    // LCD右上
        LCD.print("Hz");        // "Hz"
        LCD.setCursor(0, 1);    // LCD左下
        LCD.print("Lv");        // "Lv"
        loop_exc++;
      }
      break;
    case 2:   // 計測実行
      lowbat();               // 電池電圧低下チェックと表示 (右下)
      siglvl();               // 信号レベル表示 (左下)
      if(f_capok){            // キャプチャ(周波数確定)?
        f_capok = 0;
//      cli();                    // (!!!)
//      d = cnt_wrd;              // チェック用書き込みデータ
//      sei();                    // (!!!)
//      txprintf("%5ld\r\n", d);  // (!!!)
        d = frqcapbff();      // キャプチャデータから周波数を計算
        if(d > 99999)     d = 99999;  // max99999Hz
        f_disp  = 1;          // 表示指令
        cnt16_capchk = 0;     // キャプチャカウンタをクリア
      }
      if(cnt16_capchk >= 16){   // キャプチャ無しで1.0秒経過(16/16)
        d = 0;                  // カウント値ゼロで
        f_disp   = 1;           // 表示指令
        cnt16_capchk = 0;       // キャプチャカウンタをクリア
        f_cap1st = 0;           // キャプチャ処理最初から
        capbffclr();            // キャプチャデータバッファをクリア
      }
      if(f_disp){               // 周波数表示指令?
        f_disp = 0;             // 表示フラグクリア
        sprintf(str_bff, "%5lu", d);   // 5桁 出力データ
        LCD.setCursor(1, 0);    // LCD上段左
        LCD.print(str_bff);     // " xxxxxxHz"
      }
      if(f_1sec){         // 1秒経過
        f_1sec = 0;
        sprintf(str_bff, "%5luHz\r\n", d);   // 5桁周波数データ
        txstr(str_bff);         // シリアル出力
      }
    }
  }
}
// end of "tiny1614_fcnt_tcb04.ino"
