/*****  AQM1602Y-FLW-FBW I2C 液晶表示ライブラリ *****/
// 「LiquidCrystal」を元に改造
//	ATtiny402,202用
//  2022-07-19  setContrast,getContrast  LCD_V5R0,LCD_V3R3に
//              電源電圧設定に注意
//  2025-09-09  ライブラリ、wire,twiを使わず割り込みを用いない方法で
//				SCL周波数を設定できるように
#include "i2c_lcd_aqm1602_tn.h"
#include <inttypes.h>
#include <Arduino.h>
/********************************/
/*  ＴＷＩモジュールのアクセス  */
/********************************/
//  ATtiny402のTWIハードウェアを制御
//  割り込みを使わなない
/*****  TWI 初期化      *****/
//  F_CPUとfrq(kHz)からTWI SCL周波数を設定
//  MBAUD = F_CPU / 2 / frq - 5
void twiInit(uint16_t frq)
{
int16_t d;
    d = F_CPU / 1000;           // CPU クロック周波数(kHz)
    d = (d / 2 / frq) - 5;      // F_CPU / 2 / fSCL-5
    if(d < 0)       d = 1;      // 1～255
    if(d > 255)     d = 255;
    TWI0.MBAUD    =  d;         // TWI SCL周波数設定
    TWI0.MCTRLA   = 0b00000001;
    //                || ||||+----- ENABLE  主装置許可
    //                || |||+------ SMEN    簡便動作しない
    //                || |++------- TIMEOUT 使わない
    //                || +--------- QCEN    迅速指令 しない
    //                |+----------- WIEN    書き込み割り込み なし
    //                +------------ RIEN    読み込み割り込み なし
    TWI0.MSTATUS  = 0b00000001;
    //                ||||||++----- BUSSTATE アイドル状態に
    //                |||||+------- BUSERR
    //                ||||+-------- ARBLOST
    //                |||+--------- RXACK
    //                ||+---------- CLKHOLD
    //                |+----------- WIF
    //                +------------ RIF
}   
/*****  TWI 書込み送出開始    *****/
void twiStartW(uint8_t adrs)
{
    TWI0.MADDR = adrs << 1; // アドレス 書き込みで(0)
    while(!(TWI0.MSTATUS & TWI_WIF_bm));    // WIF=1を待つ
}
/*****  TWI データ書き込み  *****/
void twiWrite(uint8_t data)
{
    TWI0.MDATA = data;      // データ書き込み
    while(!(TWI0.MSTATUS & TWI_WIF_bm));    // WIF=1を待つ
}
/*****  TWI 送出終了    *****/
void twiStop()
{
    TWI0.MCTRLB = TWI_MCMD_STOP_gc; // Stop Condition
}

/*********************/
/*   液晶アクセス    */
/*********************/
/*****  コンストラクタ：I2Cアドレス設定     *****/
i2c_lcd_aqm1602_tn::i2c_lcd_aqm1602_tn(uint8_t adrs) {
    _i2c_adrs = adrs;               // I2Cアドレス
}

/*****  文字数、行数設定    *****/
//  液晶のイニシャル処理 文字数と行数,
//	SCL周波数,電圧を設定
//  v:電源電圧区分 0なら5V、1なら3.3V
void i2c_lcd_aqm1602_tn::begin(uint8_t  cols,   // 文字数
                               uint8_t  lines,  // 行数
                               uint16_t frq,    // SCL周波数(kHz)
                               uint8_t v) {     // 電圧(0/1)
uint8_t i;
    _numcols  = cols;               // 設定した文字数を保存
    _numlines = lines;              // 行数保存
    delayMicroseconds(20000);       // 20ms    
    twiInit(frq);                   // I2C開始 SCL周波数でMBAUD設定
    delayMicroseconds(10000);       // 10ms    
    _row_offsets[0]  = 0x00;        // 1行目　表示アドレス設定
    _row_offsets[1]  = 0x40;        // 2行目　行間は連続していない
    _row_offsets[2]  = 0x00 + cols; // 3行目　+文字数を
    _row_offsets[3]  = 0x40 + cols; // 4行目
//  コマンドデータ
    _displayfunction = 0b00111000;  // LCD機能設定 <0x38>
                //         |||| +--  0:IS 機能選択
                //         |||+----  0:8行フォント 1:11行フォント
                //         ||+-----  0:1行表示     1:2行表示 
                //         |+------  0:4bitモード  1:8bitモード
                //         +-------  0x20 コマンド
    _displaycontrol  = 0b00001100;  // 表示,カーソルon/off (0x0C>
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    _displaymode     = 0b00000110;  // エントリーモード  <0x06>  
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    if(v == LCD_V5R0){              // 電源5Vの時
      _disppower     = 0b01010000;  // Bon = 0 <0x50>
      _dispfollower  = 0b01101100;  // Foollower Fon Rbx2.0
    }
    else{                           // 電源3.3V
      _disppower     = 0b01010100;  // Bon = 1 <0x54>
      _dispfollower  = 0b01101100;  // Foollower Fon Rbx2.0
    }
    _contrast = 35;                 // コントラスト(6bit) 0～63 暫定値 ★
//  イニシャルコマンド送出 (昔は0x38を3回送っていたけれど・・・)
    command(_displayfunction);              // 機能設定 <0x38>
    command(_displayfunction | 0b00000001); // <0x39> IS=1
    command(0b00010100);            // 周波数  <0x14>
    //            |+++----- F2,1,0 192kHz(5V) 183kHz(3.3V)
    //            +-------- BS 0:bias 1/5
    command(LCD_CONTRAST | (_contrast & 0x0F));      // コントラスト下位 <0x70>
    command(_disppower | ((_contrast >> 4) & 0x03)); // コントラスト上位 <0x50>
    command(_dispfollower);         // Follower <0x6C>
    for(i = 0; i < 200; i++){       // 200ms wait
        delayMicroseconds(1000);    // 1000us
    }
    command(_displayfunction);      // 機能設定 <0x38> IS=0
    command(_displaycontrol);       // 表示 on  <0x0C>
    command(_displaymode);          // エントリーモード <0x06>
    clear();                        // 画面消去,DDRAM=0
}

/*****  表示クリア      *****/
void i2c_lcd_aqm1602_tn::clear() {
    command(LCD_CLEARDISPLAY);  // clear display, set cursor position to zero
    delayMicroseconds(2000);    // this command takes a long time!
}

/*****  カーソル原点位置へ  *****/
void i2c_lcd_aqm1602_tn::home() {
    command(LCD_RETURNHOME);    // set cursor position to zero
    delayMicroseconds(2000);    // this command takes a long time!
}

/*****  カーソル移動    *****/
//  文字コラム位置(0～cols-1) ,行(0～lines-1)
//  行位置は規制, コラム位置は規制なしで
void i2c_lcd_aqm1602_tn::setCursor(uint8_t col, uint8_t row) {
    if ( row >= _numlines ) {     // 文字行位置
         row  = _numlines - 1;    // 最下行で
    }
    command(LCD_SETDDRAMADDR | (col + _row_offsets[row]));  // DDRAMアドレス
}

/*****  表示 on/off *****/
void i2c_lcd_aqm1602_tn::noDisplay() {
    _displaycontrol &= 0b11111011;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602_tn::display() {
    _displaycontrol |= 0b00000100;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  カーソル on/off *****/
void i2c_lcd_aqm1602_tn::noCursor() {
    _displaycontrol &= 0b11111101;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602_tn::cursor() {
    _displaycontrol |= 0b00000010;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  カーソル点滅 on/off *****/
void i2c_lcd_aqm1602_tn::noBlink() {
    _displaycontrol &= 0b11111110;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602_tn::blink() {
    _displaycontrol |= 0b00000001;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  画面スクロール  *****/
void i2c_lcd_aqm1602_tn::scrollDisplayLeft(void) {
    command(0b00011000);        // カーソル移動、表示シフト
    //           ||+------  0:左シフト 1:右シフト
    //           |+-------  0:表示シフトしない 1:表示シフトする
    //           +--------  0x10 コマンド
}
void i2c_lcd_aqm1602_tn::scrollDisplayRight(void) {
    command(0b00011100);        // カーソル移動、表示シフト
    //           ||+------  0:左シフト 1:右シフト
    //           |+-------  0:表示シフトしない 1:表示シフトする
    //           +--------  0x10 コマンド
}

/*****  入力方向切替    *****/  
void i2c_lcd_aqm1602_tn::leftToRight(void) {
    _displaymode    |= 0b00000010;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}
void i2c_lcd_aqm1602_tn::rightToLeft(void) {
    _displaymode    &= 0b11111101;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}

/*****  スクロール(画面シフト) on/off   *****/
void i2c_lcd_aqm1602_tn::autoscroll(void) {
    _displaymode    |= 0b00000001;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}
void i2c_lcd_aqm1602_tn::noAutoscroll(void) {
    _displaymode    &= 0b11111110;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}

/*****  コントラスト設定 ★   *****/
//  n : 0～63 6bit
void i2c_lcd_aqm1602_tn::setContrast(uint8_t n) {
    if(n > 63)          n = 63;     // 0～63
    _contrast = n;
    command(_displayfunction | 0b00000001); // <0x39> IS=1
    command(LCD_CONTRAST |  (_contrast & 0x0F));       // コントラスト下位
    command(_disppower   | ((_contrast >> 4) & 0x03)); // コントラスト上位
    command(_displayfunction);              // <0x38> IS=0
}
/*****  コントラスト設定値読み出し ★   *****/
//  ret : 0～63 6bit
uint8_t i2c_lcd_aqm1602_tn::getContrast() {
    return _contrast;           // 0～63 現在の設定値
}

/*****  CG RAM 設定     *****/
//  location:0～7,  charmap:8x8bitのキャラジェネデータ
void i2c_lcd_aqm1602_tn::createChar(uint8_t location, uint8_t charmap[]) {
    location &= 0x7;                // we only have 8 locations 0-7
    command(LCD_SETCGRAMADDR | (location << 3));
    for (byte i = 0; i < 8; i++) {
        write(charmap[i]);      // CGデータを転送
    }
}

/*****  コマンド送出    *****/
void i2c_lcd_aqm1602_tn::command(uint8_t value) {
    twiStartW(_i2c_adrs);
    twiWrite(0x00);                   // RS = 0
    twiWrite(value);
    twiStop();
    delayMicroseconds(40);            // 40us wait
}

/*****  データ送出  *****/
size_t i2c_lcd_aqm1602_tn::write(uint8_t value) {
    twiStartW(_i2c_adrs);
    twiWrite(0x40);                   // RS = 1
    twiWrite(value);
    twiStop();
    delayMicroseconds(40);            // 40us wait
    return 1;                         // assume sucess
}

/*****  文字列表示  *****/
void i2c_lcd_aqm1602_tn::strdisp(const char *s)
{
    while(*s != '\0'){    // Nullまで
      write(*s);          // 1文字出力
      s++;                // 次文字
    }
}

// end of "i2c_lcd_aqm1602_tn.cpp"
