# st7789sync.py
from machine import Pin, SPI
import framebuf, time

# ---- 色変換/定数 ----
def fbcolor(r, g, b):
    r = (r >> 3) & 0x1F
    g = (g >> 2) & 0x3F
    b = (b >> 3) & 0x1F
    return ((g & 0x07) << 13) + (r << 8) + (b << 3) + ((g & 0x38) >> 3)

COLOR_BLACK = fbcolor(0, 0, 0)
COLOR_WHITE = fbcolor(255, 255, 255)

# ---- コマンド ----
SWRESET = 0x01;
SLPOUT = 0x11;
NORON = 0x13
DISPON = 0x29
CASET = 0x2A;
RASET = 0x2B;
RAMWR = 0x2C
MADCTL = 0x36;
COLMOD = 0x3A
MADCTL_BGR = 0x08
INVOFF = 0x20 ;
INVON = 0x21

class ST7789:
    def __init__(self, spi, width, height, reset, dc, cs, blk, x_off=0, y_off=0):
        self._spi = spi
        self._width = width
        self._height = height
        self._reset = reset; self._dc = dc; self._cs = cs; self._blk = blk
        self._x_off = x_off; self._y_off = y_off
        self._buffer = bytearray(width * height * 2)
        self._fbuf = framebuf.FrameBuffer(self._buffer, width, height, framebuf.RGB565)
        self._cs.value(1); self._dc.value(1); self._reset.value(1)
        self.reset()

    def _send_cmd(self, c):
        self._cs.value(0); self._dc.value(0); self._spi.write(bytearray([c])); self._cs.value(1)

    def _send_data(self, data):
        self._cs.value(0); self._dc.value(1)
        if isinstance(data, (bytes, bytearray, memoryview)):
            self._spi.write(data)
        else:
            self._spi.write(bytearray([data]))
        self._cs.value(1)

    def _send_cmd_data(self, c, data):
        self._send_cmd(c); self._send_data(data)

    def hw_reset(self):
        self._reset.value(0); time.sleep_ms(20)
        self._reset.value(1); time.sleep_ms(120)

    def sw_reset(self):
        self._send_cmd(SWRESET); time.sleep_ms(150)
        self._send_cmd(SLPOUT);  time.sleep_ms(120)
        self._send_cmd_data(COLMOD, b"\x55")       # 16bpp
        self._send_cmd_data(MADCTL, bytearray([MADCTL_BGR]))
        self._send_cmd(INVON);   time.sleep_ms(10)
        self._send_cmd(NORON);   time.sleep_ms(10)
        self._send_cmd(DISPON);  time.sleep_ms(100)

    def reset(self):
        self.hw_reset()
        self.sw_reset()
        self._set_window(0, 0, self._width-1, self._height-1)

    def backlight(self, on):
        self._blk.value(1 if on else 0)

    def clear(self, c):
        self._fbuf.fill(c)

    def text(self, s, x, y, c=COLOR_WHITE):
        self._fbuf.text(s, x, y, c)

    def update(self):
        self._set_window(0, 0, self._width-1, self._height-1)
        self._send_data(self._buffer)

    # ---- 矩形設定（両端含む） ----
    def _set_window(self, x0, y0, x1, y1):
        x0 += self._x_off; x1 += self._x_off
        y0 += self._y_off; y1 += self._y_off
        self._send_cmd_data(CASET, bytes([(x0>>8)&0xFF, x0&0xFF, (x1>>8)&0xFF, x1&0xFF]))
        self._send_cmd_data(RASET, bytes([(y0>>8)&0xFF, y0&0xFF, (y1>>8)&0xFF, y1&0xFF]))
        self._send_cmd(RAMWR)  # ★必ず出す

    # ---- 同期1行描画 ----
    def push_line_direct_sync(self, x, y, w, data):
        if not (0 <= x < self._width and 0 <= y < self._height):
            return
        if w <= 0:
            return
        if x + w > self._width:
            w = self._width - x
        self._set_window(x, y, x + w - 1, y)
        step = self._width * 2  # 十分大きく=分割少なめ
        mv = memoryview(data)
        for i in range(0, 2 * w, step):
            self._send_data(mv[i:i+step])

