# raytracer.py
import math

# ==================================
# 画面サイズ
# ==================================
LCD_W, LCD_H = 240, 240

# レンダリング解像度 (SCALE=2で120x120を240x240 に敷き詰め)
RENDER_W, RENDER_H = 120, 120
SCALE = 2

# オフセット(中央寄せ)
OFFX = (LCD_W - RENDER_W * SCALE) // 2
OFFY = (LCD_H - RENDER_H * SCALE) // 2

# 品質切り替え
# 既定は超軽量: SSAA_OFF=False, シャドウ無し, 反射無し
SSAA_2x2 = False   # Trueにすると2x2 SSAA
USE_SPEC  = True   # スペキュラを入れる
USE_SHADOW= False  # シャドウレイは重いので既定でOFF
USE_REFLECT=False  # 反射は再帰になるためOFF固定

# レンダリング係数（軽量寄り）
AMBIENT     = 0.20
SPEC_POWER  = 16     # 32→16にしてpow計算を軽く
SPEC_GAIN   = 0.3

# カメラ
FOV_DEG = 60.0
_f = math.tan((FOV_DEG * math.pi/180.0) * 0.5)

# =========================
# シーン定義
# =========================
# 球: (cx, cy, cz, r, (R,G,B))
SPHERES = [
    (-0.6, -0.1, 3.0, 0.45, (255,  80,  80)),  # 赤
    ( 0.6, -0.2, 2.3, 0.35, ( 80, 200, 255)),  # シアン
    ( 0.0,  0.2, 1.6, 0.25, (255, 220,  80)),  # 黄
]

# 床（無限平面）
FLOOR_Y     = -0.6
CHECKER_SZ  = 0.25
FLOOR_COL1  = (200, 200, 200)
FLOOR_COL2  = ( 50,  50,  50)

# ライト
LIGHTS = [
    { 'type': 'dir', 'dir': (-0.4, -0.7, -0.6), 'color': (255,255,255), 'intensity': 1.0 },
]

# =========================
# ベクトルユーティリティ
# =========================
def v_dot(a, b): return a[0]*b[0] + a[1]*b[1] + a[2]*b[2]
def v_sub(a, b): return (a[0]-b[0], a[1]-b[1], a[2]-b[2])
def v_len(a):    return math.sqrt(v_dot(a, a))
def v_nrm(a):
    l = v_len(a)
    if l <= 1e-12: return (0.0, 0.0, 0.0)
    inv = 1.0/l
    return (a[0]*inv, a[1]*inv, a[2]*inv)

# =========================
# 交差
# =========================
def intersect_sphere(ro, rd, sph):
    cx, cy, cz, r, col = sph
    sx, sy, sz = ro[0]-cx, ro[1]-cy, ro[2]-cz
    b = 2.0*(rd[0]*sx + rd[1]*sy + rd[2]*sz)
    c = (sx*sx + sy*sy + sz*sz) - r*r
    disc = b*b - 4.0*c
    if disc < 0.0:
        return None
    s = math.sqrt(disc)
    t1 = (-b - s)*0.5
    t2 = (-b + s)*0.5
    t = t1 if t1>1e-4 else (t2 if t2>1e-4 else None)
    if t is None:
        return None
    hit = (ro[0]+rd[0]*t, ro[1]+rd[1]*t, ro[2]+rd[2]*t)
    nrm = v_nrm((hit[0]-cx, hit[1]-cy, hit[2]-cz))
    return t, hit, nrm, col

def intersect_floor(ro, rd):
    if abs(rd[1]) < 1e-6:
        return None
    t = (FLOOR_Y - ro[1]) / rd[1]
    if t <= 1e-4:
        return None
    hit = (ro[0]+rd[0]*t, FLOOR_Y, ro[2]+rd[2]*t)
    nrm = (0.0, 1.0, 0.0)
    cx = int(math.floor(hit[0] / CHECKER_SZ))
    cz = int(math.floor(hit[2] / CHECKER_SZ))
    col = FLOOR_COL1 if ((cx+cz)&1)==0 else FLOOR_COL2
    return t, hit, nrm, col

def scene_hit(ro, rd):
    nearest = 1e9
    isect   = None
    for sph in SPHERES:
        h = intersect_sphere(ro, rd, sph)
        if h and h[0] < nearest:
            nearest = h[0]; isect = h
    f = intersect_floor(ro, rd)
    if f and f[0] < nearest:
        nearest = f[0]; isect = f
    return isect  # (t, pos, nrm, col) or None

# =========================
# シェーディング
# =========================
def shade(hit_pos, nrm, base_col):
    # 環境光
    r_acc = int(base_col[0] * AMBIENT)
    g_acc = int(base_col[1] * AMBIENT)
    b_acc = int(base_col[2] * AMBIENT)

    V = (0.0, 0.0, -1.0)  # ビュー方向（簡略固定）

    for L in LIGHTS:
        if L['type'] == 'dir':
            ld = v_nrm(L['dir'])
        else:
            # 今回は点光源は使わない
            ld_vec = v_sub(L.get('pos', (0,0,1)), hit_pos)
            ld = v_nrm(ld_vec)

        # シャドウ計算は無し
        ndotl = v_dot(nrm, ld)
        if ndotl > 0.0:
            k = ndotl * L.get('intensity', 1.0)
            r_acc += int(base_col[0] * k * (L['color'][0]/255))
            g_acc += int(base_col[1] * k * (L['color'][1]/255))
            b_acc += int(base_col[2] * k * (L['color'][2]/255))

            if USE_SPEC:
                # Blinn-Phong（軽量）：H = normalize(ld + V)
                H = v_nrm((ld[0]+V[0], ld[1]+V[1], ld[2]+V[2]))
                s = v_dot(nrm, H)
                if s > 0.0:
                    # s^SPEC_POWER （簡易 pow）
                    s = s ** SPEC_POWER
                    s *= SPEC_GAIN * L.get('intensity', 1.0)
                    r_acc += int(255 * s * (L['color'][0]/255))
                    g_acc += int(255 * s * (L['color'][1]/255))
                    b_acc += int(255 * s * (L['color'][2]/255))

    # クリップ
    if r_acc > 255: r_acc = 255
    if g_acc > 255: g_acc = 255
    if b_acc > 255: b_acc = 255
    return (r_acc, g_acc, b_acc)

# =========================
# 1サンプル or 2×2 SSAA
# =========================
SSAA_OFFSETS = (
    (-0.25, -0.25),
    ( 0.25, -0.25),
    (-0.25,  0.25),
    ( 0.25,  0.25),
)

def sample_color(i, j):
    """RENDER座標(i,j)の色（RGB888）"""
    cam = (0.0, 0.0, 0.0)
    inv_rw = 1.0 / RENDER_W
    inv_rh = 1.0 / RENDER_H

    if SSAA_2x2:
        r_acc = g_acc = b_acc = 0
        for (ox, oy) in SSAA_OFFSETS:
            sx = ((i + 0.5 + ox) * inv_rw) * 2.0 - 1.0
            sy = 1.0 - ((j + 0.5 + oy) * inv_rh) * 2.0
            rd = v_nrm((sx * _f, sy * _f, 1.0))
            h = scene_hit(cam, rd)
            if h:
                _, pos, nrm, col = h
                r, g, b = shade(pos, nrm, col)
            else:
                r, g, b = (10, 15, 20)
            r_acc += r; g_acc += g; b_acc += b
        return ((r_acc + 2) >> 2, (g_acc + 2) >> 2, (b_acc + 2) >> 2)
    else:
        # 1サンプル（高速）
        sx = ((i + 0.5) * inv_rw) * 2.0 - 1.0
        sy = 1.0 - ((j + 0.5) * inv_rh) * 2.0
        rd = v_nrm((sx * _f, sy * _f, 1.0))
        h = scene_hit(cam, rd)
        if h:
            _, pos, nrm, col = h
            return shade(pos, nrm, col)
        else:
            return (10, 15, 20)

# =========================
# RGB565（MSB→LSB）へ変換
# =========================
def _rgb888_to_565_be_bytes(r, g, b):
    r5 = (r >> 3) & 0x1F
    g6 = (g >> 2) & 0x3F
    b5 = (b >> 3) & 0x1F
    c  = (r5 << 11) | (g6 << 5) | b5
    return (c >> 8) & 0xFF, c & 0xFF

# =========================
# 公開API：LCD座標の1行を生成
# =========================
def line_rgb565_for_lcd(x0, y_lcd, w):
    """
    LCD座標 (x0..x0+w-1, y_lcd) の1行をRGB565（MSB→LSB）で返す。
    戻り値: bytearray(2*w)
    """
    out = bytearray(2 * w)

    jy = (y_lcd - OFFY)
    if jy < 0 or jy >= RENDER_H * SCALE:
        # レンダ領域外は黒
        return out
    j = jy // SCALE

    off = 0
    for dx in range(w):
        ix = (x0 + dx - OFFX)
        if ix < 0 or ix >= RENDER_W * SCALE:
            # レンダ領域外は黒
            out[off] = 0; out[off+1] = 0
            off += 2
            continue
        i = ix // SCALE

        r, g, b = sample_color(i, j)
        hi, lo = _rgb888_to_565_be_bytes(r, g, b)
        out[off]   = hi
        out[off+1] = lo
        off += 2

    return out
