/*********************************************/
/*  Arduino-UNO R3+AD9833で作った発振器      */
/*********************************************/
//  2025-05-26  "AD9833_01"
//  16bitタイマー1を使って8MHzのMCLKをOC1Aに出力。
//  CKOUTヒューズを0にして16MHzクロック出力を有効に。
//  どちらかをAD9933のMCLKとして使う。
//      #defineで定義しているMCLK_FRQ値を変更。
//  I2C液晶で表示
//  スイッチ操作
//      SW1 出力周波数 A/B切り替え
//      SW2 入力位置カーソル←
//      SW3 入力位置カーソル→
//      SW4 Sin,Tri,Sqr 波形切り替え
//      SW1長押し  EEPORMにパラメータをセーブ
//      SW2長押し  sweep開始
//      SW3長押し  sweep開始
//      SW4長押し  メニュー表示
//    ピン割り当て
//      PD0  RXD   in  RXD
//      PD1  TXD   out TXD
//      PD2  IO2   in  ロータリーエンコーダA相入力
//      PD3  IO3   in  ロータリーエンコーダB相入力
//      PD4  IO4   out Sweepトリガ信号 Rise時にH
//      PD5  IO5   out -
//      PD6  IO6   out -
//      PD7  IO7   in  コンパレータ入力(電池電圧) 
//      PB0  IO8   out 16MHz クロック出力(fuse書き換えで)
//      PB1  IO9   out OC1A 8MHz MCLK出力
//      PB2  IO10  out AD9833 FSYNC Lアクティブ
//      PB3  IO11  out AD9833 SDATA (MOSI)
//      PB4  IO12  out -
//      PB5  IO13  out AD9833 SCLK
//      PC0  AD0   in  SW1入力(Lでon) 
//      PC1  AD1   in  SW2入力
//      PC2  AD2   in  SW3入力
//      PC3  AD3   in  SW4入力
//      PC4  AD4   I2C SDA (液晶)
//      PC5  AD5   I2C SCL (液晶)
//  16MHzクロック出力 PB0:CLKO
//          ヒューズ(下位にあるCKOUT)を書き換えて(0に)
//          16MHzクロック出力を有効に//          
//  タイマー0  (システムで使っている)
//          そのままに
//  タイマー1   MCLK 8MHz方形波出力
//          AD9833のクロックに使えるように
//  タイマー2   4kHz 0.25ms割り込み 
//          計時、スイッチ入力、エンコーダ入力
//  INT0割り込み→使わない  タイマー割り込みで処理するように
//          ロータリーエンコーダA相の↓エッジを検出
//          B相はH/Lを見ているだけ
//          クリック有タイプのエンコーダを使用
//  コンパレータ入力 AIN1
//          内部基準電圧1.1Vと比較してLowBatを検出
//  EEPROM 動作設定パラメータを保存
//  液晶  16文字 x 2行
//    画面例
//           ________________ 
//          |A:  1234.1234kHz|
//          |Sin SwpTm1234.5 |
//           ---------------- 
//           0123456789012345   
//～～～～～～～～～～～～～～～～～～～～～～～～～～～～～
/*****  Include File *****/
#include <EEPROM.h>
#include "i2c_lcd_aqm1602a.h"   // LCD制御ヘッダーファイル
                                // wireライブラリは使わない
i2c_lcd_aqm1602 LCD(0x3E);      // コンストラクタ LCD I2Cアドレスを設定
                                // LCD.begin(16, 2, LCD_V5R0)でLCD初期化
                                // 文字数、行数設定 (16文字x2行) 5V電源
//  LCD文字数と行数
#define LCD_COLS    16  // 文字数
#define LCD_ROWS    2   // 行数
/*****  タイトル    *****/    // 0123456789012345
const char pgm_ttl1[] PROGMEM = "AD9833 OSC      ";
const char pgm_ttl2[] PROGMEM = "      2025-05-26";
PGM_P pgm_ttl[] = { pgm_ttl1, pgm_ttl2, };
/*****  マクロ    *****/
#define DIMSIZ(a)   (sizeof(a)/sizeof(*a))  // 配列のﾃﾞｰﾀ数を返す
//  エンコーダ入力  H/Lチェック(1でH)
#define INP_ENCA    (PIND & (1 << PD2))     // ↓エッジでカウント
#define INP_ENCB    (PIND & (1 << PD3))     // Hならup,Lならdown
//  SW入力  on,offチェック (L=onで1) 長押し判定あり
#define INP_SW1     ((~PINC) & (1 << PC0))  // 周波数 A/B切り替え
#define INP_SW2     ((~PINC) & (1 << PC1))  // カーソル←
#define INP_SW3     ((~PINC) & (1 << PC2))  // カーソル→
#define INP_SW4     ((~PINC) & (1 << PC3))  // 波形選択 Sin,Tri,Sqr
//  LowBat検出
#define CK_LOWBAT   (ACSR & (1 << ACO))     // LowBatで1
//  テスト用ポートH/L制御出力 (SBI,CBI命令に展開)
#define PD5_H       (PORTD |=  (1 << PD5))  // (!!!)PD5 H/L 11pin
#define PD5_L       (PORTD &= ~(1 << PD5))  //   OC2A割り込み 4kHz
#define PD6_H       (PORTD |=  (1 << PD6))  // (!!!)PD6 H/L 12pin
#define PD6_L       (PORTD &= ~(1 << PD6))  //   エンコーダup/down
//  Sweep時のトリガータイミング Rise時にH
#define SWPRISE_H   (PORTD |=  (1 << PD4))  // PD4 H/L 6pin
#define SWPRISE_L   (PORTD &= ~(1 << PD4))  //
//  SWコード (5,6,7,8は長押し時のコード)
#define  SW_FRQ     1       // SW1 周波数 A/B切り替え
#define  SW_CURL    2       // SW2 カーソル←
#define  SW_CURR    3       // SW3 カーソル→
#define  SW_WAVE    4       // SW4 波形選択 Sin,Tri,Sqr
#define  SW_EEP     5       // SW1長押し EEPORM書き込み
#define  SW_SWP1    6       // SW2長押し Sweep開始1
#define  SW_SWP2    7       // SW3長押し Sweep開始2
#define  SW_MENU    8       // SW4長押し メニュー表示
/*****  データ     *****/
//  表示文字バッファ
char str_bff[40];       // 文字出力,表示用データ(40文字)
                        // Serial.printでも流用
                        // sprintfを使って数値変換
//  カーソル位置 0～4 0:右端～7:左端
byte cur_pos;           // カーソル位置 (0:右端～7:右端)
                        // 周波数設定
const byte c_pos_hz[] PROGMEM = {
      12, 11, 10,  9,     // 0.1Hz～100Hz
       7,  6,  5,  4,     //  1kHz～1MHz
};
//  エンコーダデータ
int32_t enc_data;   // 32bitエンコーダデータ
                    // エンコーダ操作でup/down                    
                    // これを元にDDS周波数出力処理
byte f_disp_enc;    // エンコーダデータ表示指令フラグ
byte f_encadd;      // エンコーダデータ加算指令フラグ
                    // encadd()で処理
//  エンコーダ割り込み処理データ
volatile short cnt_add;     // エンコーダ加算値 +/-の値
volatile byte  f_cntud;     // エンコーダーup/down検出フラグ
                            // encadd()で処理
//  encadd処理での乗数 cur_posで変える
const int32_t cur_mlt[] PROGMEM = {
    1,              // 0 (右端) 0.1Hz桁
    10,             // 1        1Hz  
    100,            // 2        10Hz 
    1000,           // 3        100Hz 
    10000,          // 4        1kHz 
    100000,         // 5        10kHz 
    1000000,        // 6        100kHz 
    10000000,       // 7 (左端) 1MHz 
};
/*****  sweep用パラメータ  *****/
//  +/-するのでsignedで
//  周波数はfreq_hz[0]とfreq_hz[1]を使う
int32_t sweep_tm_rise;      // sweep上昇時間 Lo→Hi (0.1秒)
int32_t sweep_tm_fall;      // sweep下降時間 Hi→Lo
int32_t sweep_wait_hi;      // sweep 周波数Hiでの待ち時間
int32_t sweep_wait_lo;      // sweep 周波数Loでの待ち時間
/*****  数値表示関連フラグ      *****/
byte f_ch_disp;         // 出力チャンネル表示フラグ
                        // AD98_chをA: B:で左上に表示
byte f_wave_disp;       // AD9833出力波形モード表示フラグ
                        // Sin", "Tri", "Sqr"
byte f_khz_disp;        // "kHz" 単位表示フラグ
                        // 周波数の右側に表示
byte f_swpcyc_disp;     // 周波数sweepサイクル表示
                        // sweep上昇下降時間と待ち時間を合わせて表示
/************************************/
/*    AD9833制御                    */
/************************************/
//  MCLKクロック周波数
#define AD98_RESO   268435456   // AD9833 分解能 2^28
#define MCLK_FRQ    160000000   // MCLK周波数 0.1Hz単位
                                // 実周波数をセット
#define MCLK_MAX     50000000   // 最大設定周波数 0.1Hz単位
                                // MCLK_FRQの1/3～1/4程度
//  AD9833 制御データ
byte AD98_mode;         // 出力波形 0,1,2
#define M98_SIN   0     // 正弦波
#define M98_TRI   1     // 三角波
#define M98_SQR   2     // 方形波
//  周波数データ
byte AD98_ch;           // 出力チャンネル 0:A, 1:B
                        // A:FREQ0 B:FREQ1を選択
int32_t  freq_hz[2];    // ch0,ch1 設定周波数
                        // 0.1Hz単位
//  AD9833 SPI制御データ
#define SS_H      (PORTB |=  (1 << PB2))  // PB2 FSYNC信号 H/L 16pin
#define SS_L      (PORTB &= ~(1 << PB2))  //   (Lアクティブ)
#define CK_SPIF   (SPSR & (1 << SPIF))    // SPI 転送完了チェック 1で完了
/*****  AD9833 周波数レジスタ計算  *****/
//  in  hz:0.1Hz単位の周波数
//  out FREQレジスタへの設定値
int32_t hzfreq(int32_t hz)
{
int64_t d;
    d  = (int64_t)hz * (int64_t)AD98_RESO;
    d += (int64_t)(MCLK_FRQ / 2);    // 四捨五入 +0.5
    d = d / (int64_t)MCLK_FRQ;
    return (int32_t)d;
}
/*****  SPI 8bitデータ送信   *****/
//  SS出力は別に制御
void spi8(byte d)
{
    SPDR = d;               // 8bitデータ書き込み
    while(!CK_SPIF);        // 転送完了を待つ
}
/*****  SPI 16bitデータ送信 *****/
//  MSB 8bit, LSB 8bitの順
//  前後にSS制御を入れる
void spi16(word d)
{
    SS_L;                   // FSYNC=L
    spi8((byte)(d >> 8));   // 1st, MSB 8bit
    spi8((byte)d);          // 2nd, LSB 8bit
    SS_H;                   // FSYNC=H
}
/*****  AD9833 コントロールレジスタ *****/
//  D1  : MODE       0:正弦波,1:三角波
//  D3  : DIV2       0:MSB/2,1:MSB 方形波のとき
//  D5  : OPBITEN    1:方形波
//  D11 : FSELECT    FREQ0,FREQ1
//  d : 付加する制御bitデータ
void AD98ctrl(word d)
{
word a;
    switch(AD98_mode){
      case M98_SIN:     // 正弦波
        a = 0x0000;
        break;
      case M98_TRI:     // 三角波
        a = 0x0002;
        break;
      case M98_SQR:     // 方形波
        a = 0x0028;
        break;
    }
    if(AD98_ch)   a |= 0x0800;  // FSELECT ch0,ch1
    spi16(a | d);               // d:付加制御bit
}
/*****  AD9833 FREQ データセット *****/
//  ch : 0:A,1:Bチャンネル
//  frq:0.1Hz単位の周波数
void AD98freq(byte ch, int32_t frq)
{
uint32_t d;
uint16_t a;
static uint16_t m[]={   // FREQ0,1区分ヘッダー
    0x4000,     // ch0 FREQ0
    0x8000,     // ch1 FREQ1  
};
    AD98ctrl(0x2000);          // B28 on
    d = hzfreq(frq);                    // 周波数計算
    spi16(m[ch] | (d & 0x3FFF));        // LSB 14bit
    d <<= 2;                            // d >> 14のかわり
    a = (d >> 16);                      // MSBの16bitをwordに
    spi16(m[ch] | (a & 0x3FFF));        // MSB 14bit
}
/*****  AD9833 リセット  *****/
//  位相レジスタをクリア
//  周波数レジスタに初期値をセット
void AD98reset(void)
{
    spi16(0x0100);              // RESET  = 1
    spi16(0xC000);              // PHASE0 = 0
    spi16(0xE000);              // PHASE1 = 0
}

/*****************************************/
/*      EEPROMからのデータ読み出しと保存 */
/*****************************************/
//  参考:2019年4月1日：ArduinoのEEPROMアクセス、「EEMEM」について
//      http://igarage.cocolog-nifty.com/blog/2019/04/post-1bee.html
//  ※EEMEMを使うのは止めて EEPROM内データは全部longで。
//  floatもlongと同じ4バイト。
//  EEPROMのアドレスはpara_tblの順で
//  パラメータ区分
#define P_CHK       0       // EEPROM書き込みチェックデータ
#define P_BYTE2     1       // byteデータ 0,1
#define P_BYTE3     2       // byteデータ 0,1,2
#define P_CURP      3       // byteデータ カーソル位置 0～7
#define P_FRQ       4       // longデータ 周波数 1.0Hz～MCLK_MAX
#define P_SWPTM     5       // longデータ Sweep 時間(0.1秒) 
//  パラメータのアドレスと区分
struct st_para{
    const byte  typ   PROGMEM;  // データ区分
    const void  *ram  PROGMEM;  // RAMデータアドレス
};
#define EP_CK      0x6CA7       // EEPROMチェックデータ
                                // 起動時にチェックして異なっていれば初期化
const st_para para_tbl[] PROGMEM={
    {P_CHK   , NULL           },   //  0 チェックデータ
    {P_BYTE3 , &AD98_mode     },   //  1 出力波形選択 Sin,Tri,Sqr (0,1,2)
    {P_BYTE2 , &AD98_ch       },   //  2 出力チャンネル A:FREQ0 B:FREQ1選択(0,1)
    {P_CURP  , &cur_pos       },   //  3 周波数設定カーソル位置 (0～7)
    {P_FRQ,    &freq_hz[0]    },   //  4 AD9833出力周波数 ch A (0.1Hz単位)
    {P_FRQ,    &freq_hz[1]    },   //  5 AD9833出力周波数 ch B
    {P_SWPTM , &sweep_tm_rise },   //  6 sweep上昇時間 Lo→Hi (0.1秒)
    {P_SWPTM , &sweep_tm_fall },   //  7 sweep下降時間 Hi→Lo
    {P_SWPTM , &sweep_wait_hi },   //  8 sweep 周波数Hiでの待ち時間
    {P_SWPTM , &sweep_wait_lo },   //  9 sweep 周波数Loでの待ち時間
};
/*****  EEPROMパラメータ読み出し   *****/
//  okならデータをRAMにコピー   始めての起動かNGなら初期値に
//  エラーコード(okなら0000)を持ってリターン
word loadeprom(void)
{
byte i, typ;
long  d;        // データは4バイトで受け渡し
void *r;        // RAMアドレス (byte,word,long混在)
word a;         // EEPROMアドレス
word er = 0;    // EEPROMチェック結果 チェック区分をビット位置で
char s[16];     // チェック用文字列データ
//  EEPROMデータチェック   区分P_WORD0は全値ok
    for(i = 0; i < DIMSIZ(para_tbl); i++){      // loop
        typ = pgm_read_byte(&para_tbl[i].typ);  // データ区分
        a   = sizeof(long) * i;                 // EEPROMアドレス(4バイト単位)
        EEPROM.get(a, d);                       // longでリード
        switch(typ){        // データ区分
          case P_CHK:       // チェックデータ
            if(d != EP_CK)      er |= (1 << P_CHK);   // エラー発生
            break;
          case P_BYTE2:     // 0,1
            if((d < 0) || (d > 1))          er |= (1 << P_BYTE2);
            break;
          case P_BYTE3:     // 0,1,2
            if((d < 0) || (d > 2))          er |= (1 << P_BYTE3);
            break;
          case P_CURP:     // カーソル位置 0～7
            if((d < 0) || (d > 7))          er |= (1 << P_CURP);
            break;
          case P_FRQ:       // 10～MCLK_MAX  設定最大周波数
            if((d < 10)  || (d > MCLK_MAX)) er |= (1 << P_FRQ);
            break;
          case P_SWPTM:     // 0～600.0
            if((d < 0)  || (d > 6000))      er |= (1 << P_SWPTM);
            break;
        }
    }
//  エラー発生で初期値書き込み
    if(er){             // エラーあり?
        sprintf_P(str_bff, PSTR("Err:%04X"), er); // (!!!)
        Serial.println(str_bff);
        for(i = 0; i < DIMSIZ(para_tbl); i++){    // loop
          typ = pgm_read_byte(&para_tbl[i].typ);  // データ区分
          switch(typ){    // データ区分
            case P_CHK:       // チェックデータ
              d = EP_CK;
              break;
            case P_BYTE2:     // 0,1
            case P_BYTE3:     // 0,1,2
              d = 0;
              break;
            case P_CURP:      // カーソル位置 0～7
              d = 4;                // 1.0kHz位置に
              break;
            case P_FRQ:       // 10～MCLK_MAX
              d = 10000;            // 1.0kHz (0.1Hz単位)   
              break;
            case P_SWPTM:     // 0～600.0
              d = 10;               // 1.0sec  
              break;
          }
          a = sizeof(long) * i;     // EEPROMアドレス(long)
          EEPROM.put(a, d);         // long保存
        }
    }
//  データ読み出し RAMに書き込む
    for(i = 0; i < DIMSIZ(para_tbl); i++){      // loop
        typ = pgm_read_byte(&para_tbl[i].typ);  // データ区分
        r   = pgm_read_word(&para_tbl[i].ram);  // RAMアドレス
        a = sizeof(long) * i;                   // EEPROMアドレス(long)
        EEPROM.get(a, d);                       // longでリード
        switch(typ){        // データ区分
          case P_BYTE2:     // 0,1
          case P_BYTE3:     // 0,1,2
          case P_CURP:      // カーソル位置 0～7
            *(byte *)r = d;                 // byteでRAMへ書き込み
            break;
          case P_FRQ:       // 設定周波数
          case P_SWPTM:     // sweep時間
            *(long *)r = d;                 // longでRAMへ書き込み
            break;
        }
//  チェックのためにシリアル出力
        sprintf_P(str_bff, PSTR("%2d %04X:%ld"), i, a, d);  // (!!!)
        Serial.println(str_bff);
    }
    return er;          // エラーの有無
}
/*****  EEPROMパラメータ保存   *****/
//  RAM内データがbyteでもlongでEEPROMに保存
void saveeprom(void)
{
byte i, typ;
long  d;    // データは4バイトで
void *r;    // RAMアドレス
word a;     // EEPROMアドレス
//  RAMから読み出してEEPROMに書き込む
    for(i = 0; i < DIMSIZ(para_tbl); i++){      // loop
        typ = pgm_read_byte(&para_tbl[i].typ);  // データ区分
        r   = pgm_read_word(&para_tbl[i].ram);  // RAMアドレス
        switch(typ){        // データ区分
          case P_CHK:           // チェックデータ
            d = EP_CK;
            break;  
          case P_BYTE2:         // byteデータ
          case P_BYTE3:
          case P_CURP:
            d = *(byte *)r;     // RAMデータ byteで
            break;
          case P_FRQ:           // 設定周波数
          case P_SWPTM:         // sweep時間
            d = *(long *)r;     // RAMデータ longで
            break;
        }
        a = sizeof(long) * i;   // EEPROMアドレス(long)
        EEPROM.put(a, d);       // long保存
    }
}
/************************************/
/*      Sweepモードのパラメータ     */
/************************************/
/*****  Sweep設定でのパラメータ区分 *****/
#define SWP_EXIT   0       // Exit
#define SWP_SWEEP  1       // Sweep 実行
#define SWP_TM     2       // Sweep 時間(0.1秒)
//  Sweep設定パラメータデータの最小,最大値
const long swpd_min[] PROGMEM ={
        0,          // 0  Exit
        0,          // 1  Sweep 実行
        0,          // 2  Sweep 時間(0.0秒)
};
const long swpd_max[] PROGMEM ={
        0,          // 0  Exit
        0,          // 1  Sweep 実行
     6000,          // 2  Sweep 時間(600.0秒)
};
//  パラメータの区分とデータアドレス
struct st_swp{
    const byte           typ   PROGMEM;   // データ区分
    const void volatile *data  PROGMEM;  // RAMデータアドレス
};
const st_swp swp_tbl[] PROGMEM={
    {SWP_EXIT ,  NULL           },  // 0 Exit
    {SWP_SWEEP,  NULL           },  // 1 Sweep 実行
    {SWP_TM   ,  &sweep_tm_rise },  // 2 Tm Rise
    {SWP_TM   ,  &sweep_tm_fall },  // 3 Tm Fall
    {SWP_TM   ,  &sweep_wait_hi },  // 4 Wait Hi
    {SWP_TM   ,  &sweep_wait_lo },  // 5 Wait Lo
};    
//  設定メニュー                   0123456789
const char msg_menu0[]  PROGMEM = "0 Exit   ";
const char msg_menu1[]  PROGMEM = "1 Sweep  ";
const char msg_menu2[]  PROGMEM = "2 Tm Rise";
const char msg_menu3[]  PROGMEM = "3 Tm Fall";
const char msg_menu4[]  PROGMEM = "4 Wait Hi";
const char msg_menu5[]  PROGMEM = "5 Wait Lo";
PGM_P msg_menu[]={
    msg_menu0,          // 0 Exit   
    msg_menu1,          // 1 Sweep  
    msg_menu2,          // 2 Tm Rise
    msg_menu3,          // 3 Tm Fall
    msg_menu4,          // 4 Wait Hi
    msg_menu5,          // 5 Wait Lo
};

/*****  線形補間        *****/
//  map関数をfloatで
float mapf(float x,
      float in_min,  float in_max,
      float out_min, float out_max)
{
  return (x - in_min) * 
         (out_max - out_min)
         / (in_max - in_min)
         + out_min;
}
//  map関数の内部式を64bitにしてオーバーフローが無いように
long map32(long x,                  // 入力値
     long in_min,  long in_max,     // X1,X2
     long out_min, long out_max)    // Y1,Y2
{
  return ((int64_t)(x - in_min) *
          (int64_t)(out_max - out_min))
         / ((int64_t)(in_max - in_min))
         + out_min;
}
/*************************/
/*      キャラジェネ      */
/*************************/
#define CG_BATM     0x00    // 0 Bat M
#define CG_BATL     0x01    // 1 Bat L
#define CG_ARUP     0x02    // 2 ↑
#define CG_ARDN     0x03    // 3 ↓
#define CG_BARH     0x04    // 4 ￣
#define CG_BARL     0x05    // 5 ＿
#define CG_PP       0x06    // 6 Pos Pulse
#define CG_NP       0x07    // 7 Neg Pulse
/*****  キャラジェネデータ   *****/
//  コード0x00～0x07
const char cg0[] PROGMEM = {  // BAT Midle
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b11111,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b11111,    // 7 ← 8行目にもデータを入れられる
};
const char cg1[] PROGMEM = {  // BAT L
    0b01110,    // 0
    0b11011,    // 1
    0b10001,    // 2
    0b10001,    // 3
    0b10001,    // 4
    0b10001,    // 5
    0b10001,    // 6
    0b11111,    // 7
};
const char cg2[] PROGMEM = {    // ↑
    0b00100,    // 0
    0b01110,    // 1
    0b10101,    // 2
    0b00100,    // 3
    0b00100,    // 4
    0b00100,    // 5
    0b00100,    // 6
    0b00000,    // 7
};
const char cg3[] PROGMEM = {    // ↓
    0b00100,    // 0
    0b00100,    // 1
    0b00100,    // 2
    0b00100,    // 3
    0b10101,    // 4
    0b01110,    // 5
    0b00100,    // 6
    0b00000,    // 7
};
const char cg4[] PROGMEM = {    // ￣
    0b11111,    // 0
    0b11111,    // 1
    0b00000,    // 2
    0b00000,    // 3
    0b00000,    // 4
    0b00000,    // 5
    0b00000,    // 6
    0b00000,    // 7
};
const char cg5[] PROGMEM = {    // ＿
    0b00000,    // 0
    0b00000,    // 1
    0b00000,    // 2
    0b00000,    // 3
    0b00000,    // 4
    0b11111,    // 5
    0b11111,    // 6
    0b00000,    // 7
};
const char cg6[] PROGMEM = {  // Pos Pulse
    0b00000,    // 0
    0b01110,    // 1
    0b01010,    // 2
    0b01010,    // 3
    0b01010,    // 4
    0b01010,    // 5
    0b11011,    // 6
    0b00000,    // 7
};
const char cg7[] PROGMEM = {  // Neg Pulse
    0b00000,    // 0
    0b11011,    // 1
    0b01010,    // 2
    0b01010,    // 3
    0b01010,    // 4
    0b01010,    // 5
    0b01110,    // 6
    0b00000,    // 7
};
PGM_P P_cg_tbl[]={
    cg0, cg1, cg2, cg3, cg4, cg5, cg6, cg7, // 8データ
};

/*****  キャラジェネ設定    *****/
//  8つのCGデータを設定
void lcdcgset(void)
{
byte bff[8];
byte i;
    for(i = 0; i < DIMSIZ(P_cg_tbl); i++){        // 8データ
        memcpy_P(bff, P_cg_tbl[i], sizeof(bff));  // CGデータをコピー
        LCD.createChar(i, bff);             // キャラジェネ設定  
    }
}

/************************/
/*    表示処理          */
/************************/
/*****  書式付液晶表示      *****/
//  PSTR("%d")で書式指定
void LCDprintf_P(const char *s, ...) {
va_list vp;
    va_start(vp, s);
    vsnprintf_P(str_bff, sizeof str_bff, s, vp);
    LCD.print(str_bff);
    va_end(vp);
}

/*****  AD9833出力チャンネル表示  *****/
void chdisp(void)
{
static const char *s[] = { "A: ",  "B: ", };  // ch A: B:
    if(f_ch_disp){
        f_ch_disp = 0;
        LCD.noCursor();           // いったんカーソルオフ
        LCD.setCursor(0, 0);      // 左上
        LCD.print(s[AD98_ch]);    // クロック周期表示
        enccuron();               // エンコーダー入力位置カーソル表示
    }
}
/*****  AD9833出力波形モード表示  *****/
//  Sin, Tri, Sqr
void wavedisp(void)
{
static const char *s[] = { "Sin", "Tri", "Sqr", };  // ch A: B:
    if(f_wave_disp){
        f_wave_disp = 0;
        LCD.noCursor();           // いったんカーソルオフ
        LCD.setCursor(0, 1);      // 左下
        LCD.print(s[AD98_mode]);  // 波形モード表示
        enccuron();               // エンコーダー入力位置カーソル表示
    }
}
/*****  AD9833出力周波数 kHz単位表示  *****/
//  kHz単位表示
void khzdisp(void)
{
    if(f_khz_disp){
        f_khz_disp = 0;
        LCD.noCursor();           // いったんカーソルオフ
        LCD.setCursor(13, 0);     // 右上
        LCD.print("kHz");         // 波形モード表示
        enccuron();               // エンコーダー入力位置カーソル表示
    }
}
/*****  周波数sweepサイクル表示  *****/
void swpcycdisp(void)
{
byte i;
int32_t d;          // sweepトータル時間(0.1秒単位)
static const int32_t *p[] = {      // スイープ時間
    &sweep_tm_rise,      // sweep上昇時間 Lo→Hi (0.1秒)
    &sweep_tm_fall,      // sweep下降時間 Hi→Lo
    &sweep_wait_hi,      // sweep 周波数Hiでの待ち時間
    &sweep_wait_lo,      // sweep 周波数Loでの待ち時間
};
    if(f_swpcyc_disp){
        f_swpcyc_disp = 0;
        d = 0;              // トータル時間(0.1秒単位)
        for(i = 0; i < DIMSIZ(p); i++){
            d += *p[i];
        }
        LCD.noCursor();           // いったんカーソルオフ
        LCD.setCursor(4, 1);      // 左下
        LCDprintf_P(PSTR("SwpTm%4ld.%1ld"),
                d / 10,           // 秒値
                d % 10);          // 0.1秒
        enccuron();               // エンコーダー入力位置カーソル表示
    }
}
/*****  エンコーダ設定位置カーソル表示on/off *****/
byte f_enccuroff;         // 1をセットすると
                          // enccuronをスキップ カーソルなしに
/*****  エンコーダ設定位置へカーソル表示 *****/
//  ロータリースイッチでの入力する位置を表示
//  cur_pos:0～7の8桁 0が右端
//  小数点が間に入る
void enccuron(void)
{
const byte *p;
    if(f_enccuroff == 0){         // カーソル表示する
      p = &c_pos_hz[cur_pos];             // カーソルX位置選択
      LCD.setCursor(pgm_read_byte(p), 0); // 1行目
      LCD.cursor();               // カーソル位置文字の下段に[＿]
    }
    else{                         // カーソル表示しない
        LCD.noCursor();           // カーソルオフ
        LCD.setCursor(15, 1);     // 右下 lowbatの位置に
    }
}

/******************************/
/*      スイッチ入力          */
/******************************/
//  短押し→長押し変換
const byte sw_hold[] PROGMEM = {
        0,              // off
        SW_EEP,         // SW1 長押し EEPORM書き込み
        SW_SWP1,        // SW2 長押し Sweep開始
        SW_SWP2,        // SW3 長押し Sweep開始
        SW_MENU,        // SW4 長押し メニュー表示
};
//  スイッチチェックデータ
volatile byte  f_swon;      // SW入力オン確定フラグ
volatile byte  sw_code;     // SW on コード
/*****  スイッチ入力      *****/
//  onでL, 1～4のSWコードで返す
//  長押しの判断でコードを変える
byte swinp(void)
{
    if(INP_SW1)   return  SW_FRQ;  // 周波数 A/B切り替え
    if(INP_SW2)   return  SW_CURL;  // カーソル←
    if(INP_SW3)   return  SW_CURR;  // カーソル→
    if(INP_SW4)   return  SW_WAVE;   // 波形選択 Sin,Tri,Sqr
    return 0;                       // 4つともoff
}
/*****  スイッチ入力チェック      *****/
//  1msタイマー割り込みで処理
//  結果をf_swonとsw_codeに残す
//  0.8秒経過で長押し
void swscan(void)
{
byte d;
static byte dx;             // 前回SWデータ
static byte  chk = 0;       // 実行区分
static word  tm1 = 0;       // 1msタイマー チャタリング除去
    if(tm1)             tm1--;      // 1ms計時
    d = swinp();                    // SW入力 1～3のSWコード
    switch(chk){
      case 0:      // オフチェックタイマーをセット
        tm1 = 10;                   // 10ms
        chk++;
        break;
      case 1:      // 離されるのを待つ
        if(d == 0){                 // 全オフ確認
          if(tm1 == 0)    chk++;    // タイムアップで次stepへ
        }
        else{                       // どれか押されている
          chk--;                    // タイマー再セット
        }
        break;
      case 2:      // どれかon待ち
        if(d != 0){                 // on?
          dx  = d;                  // SWコードを保存
          tm1 = 10;                 // 10ms チャタリング除去
          chk++;                    // 安定チェックへ
        }
        break;
      case 3:    // on安定待ち
        if(dx != d){                // 異なればもういちど
          chk--;
        }  
        else{                       // 安定
          if(tm1 == 0){             // タイムアップでSW確定
            tm1 = 800;              // 0.8秒経過で長押し
            chk++;                  // 長押しの判断へ
          }
        }
        break;
      case 4:   // 短押し、長押しの判断
        if(d == dx){                // on継続
          if(tm1 == 0){             // タイムアップ =長押し
            sw_code = pgm_read_byte(&sw_hold[d]);  // 長押しコード確定
            f_swon  = 1;            // onフラグ
            chk = 0;                // オフ確認に
          }
        }
        else{                       // 離された =短押し
            sw_code = dx;           // 短押しコード確定
            f_swon  = 1;            // onフラグ
            chk = 0;                // オフ確認に
        }
        break;
    }
}
/*****  SW オンチェック   *****/
//  onしたらコードを持ってリターン
//  offなら0
byte swonchk(void)
{
    if(f_swon == 0){
         return 0;              // SW off =0
    }
    else{
        f_swon = 0;             // フラグをオフして
        return sw_code;         // SW on =SWコード
    }
}
/******************************/
/*      エンコーダ入力        */
/******************************/
//  エンコーダカウント値加速データ
//  前回有効パルスからの経過時間で加速値を設定
//  チャタリング除去して2ms経過で有効パルス
const byte enc_spd[] PROGMEM = {
   100,         //  0 ms (無効パルス)
   100,         //  1 ms (  〃      )
   100,         //  2 ms
   100,         //  3 ms
    60,         //  4 ms
    40,         //  5 ms
    30,         //  6 ms
    20,         //  7 ms
    15,         //  8 ms
    10,         //  9 ms
     9,         // 10 ms
     8,         // 11 ms
     7,         // 12 ms
     6,         // 13 ms
     5,         // 14 ms
     5,         // 15 ms
     4,         // 16 ms
     4,         // 17 ms
     3,         // 18 ms
     3,         // 19 ms (20ms以上は2  50ms以上は1)
};                
//  エンコーダデータ
//  bit0とbit1:にエンコーダ入力データが入る
//    A相  B相
volatile byte enc_sft[4];   // チャタリング除去用シフトデータ
                            // 4回ともH/L同じなら安定
volatile byte enc_wrp;      // シフトレジスタ書き込み位置 0～3
volatile byte enc_old;      // 前回の確定値
volatile byte enc_new;      // 入力のon/off安定状態
volatile byte tm_enc;       // エンコーダーチェックタイマー
volatile byte tm_encacc;    // エンコーダーアクセル処理タイマー
                            // 入力処理の中で経過時間をチェック
                            // 4kHz(0.25ms)でカウントアップ
/*****    エンコーダ入力スキャン      *****/
//  タイマー割り込みで呼出
//  4サイクルでチャタリング除去
//  結果をenc_nowにセット
void encscan(void)
{
volatile byte d;        // ポート入力データ
volatile byte d0, d1;   // L安定,H安定データ
volatile byte a;        // 安定データ
volatile short add;     // カウント加算値
//  エンコーダA,B相入力
    d = 0;
    if(INP_ENCA)    d |= 0b00000001;    // bit 0:A相 on/off
    if(INP_ENCB)    d |= 0b00000010;    // bit 1:B相 
    enc_sft[enc_wrp] = d;               // 最新エンコーダ入力状態
    enc_wrp++;                                      // 次書き込み位置
    if(enc_wrp >= DIMSIZ(enc_sft))    enc_wrp = 0;  // ポインタ一周
//  シフトデータをANDとORして変化点を見つける
    d0  = enc_sft[0];           // d0:全Lチェック
    d0 |= enc_sft[1];           // [0]～[3]の4シフトデータ
    d0 |= enc_sft[2];
    d0 |= enc_sft[3];
    d1  = enc_sft[0];           // d1:全Hチェック
    d1 &= enc_sft[1];
    d1 &= enc_sft[2];
    d1 &= enc_sft[3];
    a  = (~d0) | d1;            // 1のところが安定データ
                                // 0はチャタリング中    
    enc_old = enc_new;          // 前の状態
    enc_new = (enc_old & (~a)) | (d & a);   // 新入力状態確定
                                // チャタリングのあるところは前のデータ
//  A相↓エッジチェック
//  B相がHならカウントアップ
//       Lならカウントダウン
    if((~enc_new & enc_old & 1) &&      // A↓エッジ(bit 0)
       (tm_enc >= (2*4))){              // 前のパルスから2ms経過で新↓エッジ
        PD6_H;                          // (!!!)
        if(tm_enc >= (50*4))      add = 1;     // 50ms経過してたら+1
        else if(tm_enc >= (20*4)) add = 2;     // 50～20msなら+2
        else{                                  // 19ms内はテーブルで
            add = (short)pgm_read_byte(&enc_spd[tm_enc/4]); // add値
        }
        if(enc_new & 2) cnt_add += add; // B相 = H, カウントアップ
        else            cnt_add -= add; // B相 = L, カウントダウン
        f_cntud = 1;                    // カウント値更新
        tm_enc = 0;                     // タイマー1クリアー
    }
    else{                               // エッジなし
        PD6_L;                          // (!!!)
    }
}
/***************************/
/*    タイマー処理         */
/***************************/
//  タイマーデータ
volatile byte tm_1ms;       // 1msダウンカウントタイマー
volatile byte tm_10ms;      // 10msダウンカウントタイマー
volatile byte tm_lowbat;    // LowBatチェックタイマー
                            // 10msでダウンカウント
volatile word tm_sweep;     // sweep時間 (word処理)
                            // 10msでダウンカウント
volatile byte f_10ms;       // 10ms経過でオン
                            // sweepするタイミング
volatile byte f_100ms;      // 100ms経過でオン
                            // sweep表示で使用
/*****  タイマー2 : 4kHz 0.25ms割り込み *****/
//  10msを計時
//  エンコーダ入力、スイッチ入力も行う
//  16MHz/32/125=4kHz=0.25ms
ISR(TIMER2_COMPA_vect)
{
static byte cnt_4;          // 1msカウント
static byte cnt_10;         // 10msカウント
static byte cnt_100;        // 100msカウント
    PD5_H;                  // (!!!) 11pin
//  0.25ms
    encscan();              // エンコーダ入力
    if(tm_enc != 255)     tm_enc++;     // タイマ+1 max 255
    if(tm_encacc != 255)  tm_encacc++;  // アクセル処理用
                                        // 4kHzで64ms
//  1ms
    cnt_4++;                // 0.25msごとに+1
    if(cnt_4 >= 4){         // 1ms経過
      cnt_4 = 0;
//  スイッチ入力 1ms
      if(tm_1ms)        tm_1ms--;   // 1msタイマー ダウンカウント
      swscan();                     // スイッチ入力
//  10ms
      cnt_10++;             // 10ms計時
      if(cnt_10 >= 10){
        cnt_10 = 0;
        if(tm_10ms)     tm_10ms--;    // 10msタイマー ダウンカウント
        if(tm_lowbat)   tm_lowbat--;  // LowBat表示用タイマー
        if(tm_sweep)    tm_sweep--;   // sweep時間
        f_10ms = 1;                   // 10msタイミング
//  100ms
        cnt_100++;                    // 100ms計時
        if(cnt_100 >= 10){
          cnt_100 = 0;
          f_100ms = 1;                // 100msタイミング
        }
      }
    }
    PD5_L;                  // (!!!)
}
/*****  LowBat表示        *****/
//  LowBatならCG_BATL,CG_BATMを右下に点滅
//  okならspaceに
//  1秒サイクルでチェック
void lowbat(void)
{
char c;
static byte blk = 0;
    if(tm_lowbat == 0){         // タイムアップ
      blk ^= 1;                 // 点滅フラグ反転
      tm_lowbat = 100;          // タイマー1秒セット
      if(CK_LOWBAT == 0)    c = ' ';        // BAT OK,spaceに
      else{                                 // Low BAT検出
        if(blk == 0)        c = CG_BATL;    // Lマーク
        else                c = CG_BATM;    // Mマーク
      }
      LCD.noCursor();           // いったんカーソルオフ
      LCD.setCursor(15, 1);     // LCD 下段右
      LCD.write(c);             // 下段右に電池マーク
      enccuron();               // エンコーダー入力位置カーソル表示
    }
}
/*****  10ms待ち      *****/
//  n=255で2.55秒 n=100で1秒
//  スイッチ,エンコーダ操作でbreak
void wait10ms(byte n)
{
    tm_10ms = n;            // タイマーセット
    while(tm_10ms){         // タイムアップまでwait
        lowbat();           // LowBat表示実行
        if(f_swon ||        // SW on?
           f_cntud){        // エンコーダ操作?
            break;          // 時間待ち中止
        }
    }
}
/*****  操作スイッチオフ待ち  *****/
void waitswoff(void)
{
    while(swinp()){         // SWオフ待ち
        lowbat();           // LowBat表示実行
    }
}
/*****  操作スイッチオン待ち  *****/
void waitswon(void)
{
    while(swonchk() == 0){    // SWオン待ち
        lowbat();             // LowBat表示実行
    }
}
/********************************/
/*      エンコーダー処理        */
/********************************/
/*****  エンコーダーA相↓エッジ割り込み *****/
//  INT0割り込み 
//  A相↓エッジ, B相がHならカウントアップ
//               Lならカウントダウン
//  結果をカウント加算値cnt_addに残す
//  40msで加速判定 ±1か±2
#if 0   // タイマー割り込みでエンコーダを処理するので使わない
ISR(INT0_vect)
{
volatile short d;
    if((tm_enc >= 4) &&                 // 前のパルスから4ms経過で新↓エッジ
       (INP_ENCA == 0)){                // A相がHならミストリガ
        if(tm_enc >= 50)       add = 1;    // 50ms経過してたら+1
        else if(tm_enc >= 20)  add = 2;    // 50～20msなら+2
        else                   add = 4;    // 20ms内なら+4
        if(INP_ENCB)    cnt_add += d;   // inB=H カウントアップ
        else            cnt_add -= d;   // inB=L カウントダウン
        f_cntud = 1;                    // カウント値更新
        tm_enc = 0;                     // タイマー1クリアー
    }
}
#endif
/*********************************************/
/*      エンコーダup/downと表示,周波数設定   */
/*********************************************/
/*****  エンコーダ加算処理  *****/
//  f_cntudでエンコーダからのパルス有
//  f_encaddオンでenc_dataにcnt_addを加算
//  cur_posで加算桁位置を移動
//      0:1の桁 1:10の桁 ～ 4:10000の桁
void encadd(void)
{
int32_t a;      // 加算値 cnt_add (エンコーダ割り込みから)
int32_t d;      // 現在データ(freq_hz)
static int32_t m = MCLK_MAX;        // 設定最大周波数
static int32_t n = 10;              // 周波数min値 =1.0Hz
    if(f_cntud){                    // エンコーダ割り込み有?
        f_cntud  = 0;               // フラグクリア
        f_encadd = 1;               // cnt_add加算処理指令フラグをon
    }
//  エンコーダデータ加算処理
    if(f_encadd){                   // 加算処理?
        cli();                      // いったん割込禁止
        a = (int32_t)cnt_add;       // 加算値を32bit値に
        cnt_add  = 0;               // ゼロクリアして次を待つ
        f_encadd = 0;               // 指令フラグをクリア
        sei();                      // 割込再開
        a *= pgm_read_dword(&cur_mlt[cur_pos]);    // カーソル位置での乗数
        d = freq_hz[AD98_ch];       // 出力周波数(0.1Hz単位) ch A,B
        d += a;                     // 新パルス値を加算
//  カーソル位置で最大,最小を判断
        if(cur_pos == 0){                 // カーソル1桁目 最小桁+/-
          if(d > m)         d = m;        // max
          if(d < n)         d = n;        // min 範囲で規制
          freq_hz[AD98_ch] = d;           // 出力周波数確定
        }
        else{                       // カーソル2桁目より左
          if((d <= m) &&            // max okで
             (d >= n)){             // min値もok?
              enc_data = (word)d;   // okで16bit値をセーブ
            freq_hz[AD98_ch] = d;   // 出力周波数確定
          }
        }
//  データ表示へ
        f_disp_enc = 1;         // データ表示指令,PWM設定指令
    }
}
/*****  周波数表示  *****/
//  frq値を12345.1234kHzで表示
//  最小桁は0.1Hz  最上位は10MHz
void frqdisp(int32_t frq)
{
      LCD.noCursor();           // いったんカーソルオフ
      LCD.setCursor(3, 0);      // 3コラム目から
      sprintf_P(str_bff, PSTR("%5ld.%04ld"),
                   frq / 10000,       // 整数部
                   frq % 10000);      // 小数部 0.1Hzが最小桁
      LCD.print(str_bff);             // 12345.1234kHz表示
}

/*****  エンコーダデータ表示  *****/
//  freq_hzを表示
void encdisp(void)
{
int32_t d1, d2;
    if(f_disp_enc){             // 表示指令あり?
      f_disp_enc = 0;
      frqdisp(freq_hz[AD98_ch]);    // AD9833 ch A,Bで区分
      enccuron();               // エンコーダー入力位置カーソル表示
    }
}

/*****************************/
/*      パラメータ表示       */
/*****************************/
/*****  パラメータ設定データ  *****/
// J_PARAで使用
byte para_typ;          // パラメータ種別
long para_data;         // 入力中のパラメータデータ
long para_min;          // パラメータの最小値
long para_max;          // パラメータの最大値
long *para_ptr;         // 保存先のアドレス
char para_str[10];      // パラメータ設定値表示用
                        // 液晶表示で使用
/*****  パラメータ数値を表示文字列に   *****/
//  para_dataを6文字で para_typで区分
void paradstr(void)
{
char s[10];
    switch(para_typ){   // パラメータ種別
      case SWP_EXIT:    // EXIT
      case SWP_SWEEP:   // Sweep 実行
        strcpy_P(para_str, PSTR("      "));     // 6文字space
        break;
      case SWP_TM:      // 時間 0.0～600.0s
        sprintf_P(para_str, PSTR("%3ld.%1lds"),
                para_data / 10, para_data % 10 );
        break;
    }
}
/*****  パラメータ読み出し  *****/
//  n : データ番号
//  para_strに文字列としてセット
void paradread(byte n)
{
    para_typ = pgm_read_byte(&swp_tbl[n].typ);   // データ区分
    para_ptr = pgm_read_word(&swp_tbl[n].data);  // データアドレス
    switch(para_typ){
      case SWP_TM:
        para_data = *para_ptr;          // longデータ読み出し
        break;
    }
    paradstr();              // パラメータデータを文字に
}
/*****  パラメータ書き込み  *****/
//  すでにpara_typとpara_ptrはセットされている
void paradwrite(void)
{
    switch(para_typ){
      case SWP_TM:
        *para_ptr = para_data;          // longデータ書き込み
        break;
    }
}

/*****************************/
/*      メイン実行処理       */
/*****************************/
//  実行区分
byte j_exc;                 // 実行モード
#define J_PULSE      3      // パルス出力
#define J_MENU       5      // メニュー選択
#define J_PARA       7      // パラメータ設定
#define J_SWEEP     10      // Sweep実行
#define J_SWEEP1    11      // Sweep データセット
#define J_SWEEP3    13      // Sweep 次実行
//  メニュー表示用カーソル位置
byte menu_cp;           // カーソル位置 (データ入力位置)
byte menu_dp;           // 表示先頭位置

/*****************************/
/*      タイトル表示          */
/*****************************/
/*****  タイトル表示      *****/
void jttl(void)
{
byte i;
    f_enccuroff = 1;        // エンコーダ入力位置カーソル表示なし
    for(i = 0; i < DIMSIZ(pgm_ttl); i++){  // タイトル 2行
        strcpy_P(str_bff, pgm_ttl[i]);
        LCD.setCursor(0, i);
        LCD.print(str_bff);         // 液晶表示
        Serial.println(str_bff);    // シリアル出力   
    }
    tm_10ms = 200;              // 2秒
    j_exc++;
}
/*****  タイトル表示 #1       *****/
//  EEPROMからパラメータを読み出してチェック
void jttl1(void)
{
word er;
    lowbat();               // LowBat表示実行
    if(tm_10ms == 0){       // タイムアップ
        LCD.clear();        // 液晶画面消去
        er = loadeprom();   // EEPROMパラメータ読み出し
        if(er){             // エラーあり
            LCD.clear();            // 液晶画面消去
            LCD.setCursor(0, 1);    // 下段にイニシャルしたを表示
            LCD.print(F("* Init EEPROM *"));  // 液晶表示
            tm_10ms = 200;                    // 2秒
        }
        j_exc++;            // next
    }
}
/*****  タイトル表示 #2       *****/
void jttl2(void)
{
    lowbat();               // LowBat表示実行
    if(tm_10ms == 0){       // タイムアップ
        j_exc++;            // next
    }
}

/*****************************/
/*      パルス出力           */
/*****************************/
//  SW操作でCW,CCWパルスを出力
/*****  パルス出力      *****/
//  エンコーダ入力をクリア
void jpulse(void)
{
    LCD.clear();            // 液晶画面消去
    LCD.noBlink();          // カーソル点滅なし
//  エンコーダ入力をクリア
    cli();                  // いったん割込禁止
    f_cntud = 0;            // カウントパルスなしに
    cnt_add = 0;            // 加算値ゼロクリア
    sei();                  // 割込再開
//  表示指令
    f_swon        = 0;      // SWの再押しを待つ
    f_enccuroff   = 0;      // エンコーダ入力位置カーソル表示あり
    f_disp_enc    = 1;      // エンコーダデータ周波数表示
    f_ch_disp     = 1;      // AD9833出力チャンネル表示
    f_wave_disp   = 1;      // AD9833出力波形表示
    f_khz_disp    = 1;      // 周波数単位表示
    f_swpcyc_disp = 1;      // スイープサイクル時間表示
    j_exc++;                // next
}
/*****  パルス出力      *****/
//  エンコーダで周期を設定
void jpulse1(void)
{
byte *cp;   // カーソル位置
    lowbat();               // LowBat表示実行
    encadd();               // エンコーダ カウント値加算
    if(f_disp_enc){
      encdisp();              // エンコーダ値表示
      AD98freq(AD98_ch, freq_hz[AD98_ch]);  // 周波数出力 ch A/B
    }
    chdisp();                 // AD9833出力チャンネル表示
    wavedisp();               // AD9833出力波形表示
    khzdisp();                // 周波数単位表示
    swpcycdisp();             // スイープサイクル時間表示
//  スイッチチェック    長押しで別動作
    switch(swonchk()){  // スイッチ入力
      case SW_FRQ:      // SW1 周波数 A/B切り替え
        AD98_ch++;                      // 0:A 1:B
        if(AD98_ch > 1)    AD98_ch = 0; 
        f_disp_enc    = 1;      // エンコーダデータ周波数表示
        f_ch_disp     = 1;      // AD9833出力チャンネル表示
        break;
      case SW_CURL:     // SW2 カーソル←
        if(cur_pos == 7)    cur_pos =  0;   // 左端なら右端へ
        else                cur_pos += 1;   // 左へ
        enccuron();                   // encカーソル位置表示
        break;
      case SW_CURR:     // SW3 カーソル→
        if(cur_pos == 0)    cur_pos =  7;   // 右端なら左端へ
        else                cur_pos -= 1;   // 右へ
        enccuron();                   // encカーソル位置表示
        break;
      case SW_WAVE:     // SW4 波形選択 Sin,Tri,Sqr
        AD98_mode++;
        if(AD98_mode > M98_SQR)   AD98_mode = 0;
        AD98ctrl(0);                // 出力波形変更
        f_wave_disp   = 1;          // AD9833出力波形表示
        break;
      case SW_EEP:      // SW1長押し EEPORM書き込み
        LCD.noCursor();               // カーソルオフ
        LCD.setCursor(0, 1);          // 下行
        LCD.print(F("Save param.     "));
                //   01234567890012345
        saveeprom();                  // パラメータ保存
        waitswoff();                  // SWオフを待つ
        wait10ms(100);                // 1秒待ち
        j_exc = J_PULSE;              // パルス出力の先頭に戻る
        break;
      case SW_SWP1:     // SW2長押し Sweep開始
      case SW_SWP2:     // SW3長押し Sweep開始
        j_exc = J_SWEEP;            // Sweep開始
        break;
      case SW_MENU:        // SW4長押し メニュー表示
        j_exc = J_MENU;             // メニュー表示へ
        break;
    }
}

/***********************/
/*      メニュー       */
/***********************/
/*****  メニュー    *****/
//  2行に表示
void jmenu(void)
{
    LCD.clear();                            // 液晶画面消去
    LCD.noCursor();                         // カーソルなし
    LCD.setCursor(0, 0);                    // 1行目
    strcpy_P(str_bff, msg_menu[menu_dp]);
    LCD.print(str_bff);                     // 液晶表示
    LCD.setCursor(10, 0);                   // 上段中央
    paradread(menu_dp);
    LCD.print(para_str);                    // 現在値表示
//  2行目
    LCD.setCursor(0, 1);                    // 2行目
    strcpy_P(str_bff, msg_menu[menu_dp+1]);
    LCD.print(str_bff);                     // 液晶表示
    LCD.setCursor(10, 1);                   // 下段中央
    paradread(menu_dp+1);
    LCD.print(para_str);                    // 現在値表示
//  カーソル
    LCD.setCursor(1, menu_cp - menu_dp);    // カーソル位置
    LCD.blink();                            // カーソル点滅
    j_exc++;                        // 次exc
}
/*****  メニュー #1   *****/
void jmenu1(void)
{
static const byte j[] PROGMEM = {   // j_exc設定番号
    J_PULSE,         // 0 Exit   
    J_SWEEP,         // 1 Sweep  
    J_PARA,          // 2 Tm Rise
    J_PARA,          // 3 Tm Fall
    J_PARA,          // 4 Wait Hi
    J_PARA,          // 5 Wait Lo
};
short d;
    switch(swonchk()){
      case SW_FRQ:     // SW1 (短押し)
        j_exc = J_SWEEP;            // Sweep開始
        break;
      case SW_WAVE:    // SW4 (短押し)
        j_exc = pgm_read_byte(&j[menu_cp]);  // それぞれへ
        break;
      case SW_SWP1:    // SW2長押し
      case SW_SWP2:    // SW3長押し
      case SW_MENU:    // SW4長押し
        j_exc = J_PULSE;             // パルス出力に戻る
        break;
      case SW_EEP:      // SW1長押し パラメータセーブ
        saveeprom();                // EEPROMにパラメータ保存
        break;
      case SW_CURL:     // SW2 カーソル←
        if(menu_cp > 0){            // カーソルが先頭以外の時
            if(menu_cp == menu_dp){
                menu_dp--;          // 表示先頭を上へ
            }
            menu_cp--;              // カーソルを上へ
            j_exc--;
        }
        break;
      case SW_CURR:     // SW3 カーソル→
        if(menu_cp < (DIMSIZ(msg_menu) - 1)){   // 最終行以外の時
            if((menu_cp - menu_dp) >= (LCD_ROWS - 1)){
                menu_dp++;          // 表示先頭を下へ
            }
            menu_cp++;              // カーソル位置を下へ
            j_exc--;
        }
        break;
    }
//  エンコーダup/down入力
    if(f_cntud){                // エンコーダup/downあり?
        cli();                  // 割り込み禁止で読み出し
        d = cnt_add;            // エンコーダup/down値
        cnt_add  = 0;           // ゼロクリアして
        f_cntud = 0;            // 次入力待ちに
        sei();                  // 割込有効に戻す
        if(d > 0){          // +
            f_swon  = 1;        // sw onに
            sw_code = SW_CURR;  // カーソル→を押したことに
        }
        else if(d < 0){     // -
            f_swon  = 1;        // sw onに
            sw_code = SW_CURL;  // カーソル←を押したことに
        }
    }
}
/********************************/
/*      パラメータ設定          */
/********************************/
/*****  パラメータ設定     *****/
void jpara(void)
{
    paradread(menu_cp);                     // 現在値
    para_min = pgm_read_dword(&swpd_min[para_typ]);    // min値
    para_max = pgm_read_dword(&swpd_max[para_typ]);    // max値
//  エンコーダ入力クリアしておく
    cli();                  // 割り込み禁止
    cnt_add  = 0;           // ゼロクリアして
    f_cntud = 0;            // 次入力待ちに
    sei();                  // 割込有効に戻す
    j_exc++;
}
/*****  パラメータ設定 #1  *****/
//  カーソル表示
void jpara1(void)
{
    LCD.setCursor(9, menu_cp - menu_dp);    // カーソル中央
    LCD.blink();                            // カーソル点滅
    j_exc++;
}
/*****  パラメータ設定 #2  *****/
//  数値変更
void jpara2(void)
{
short d;
    switch(swonchk()){
      case SW_FRQ:        // SW1(短押し)
      case SW_SWP1:       // SW2(長押し)
      case SW_SWP2:       // SW3(長押し)
        paradwrite();               // 入力パラメータ保存
        j_exc = J_SWEEP;            // Sweep開始
        break;
      case SW_WAVE:       // SW4(短押し)
        paradwrite();               // 入力パラメータ保存
        j_exc = J_MENU;             // メニューヘ
        break;
      case SW_EEP:        // SW1(長押し) パラメータセーブ
        paradwrite();               // 入力パラメータ保存
        saveeprom();                // EEPROMにパラメータ保存
        break;
      case SW_CURL:       // SW2 カーソル←
      case SW_CURR:       // SW3 カーソル→
        j_exc = J_MENU;             // 保存せずメニューに戻る
        break;
      case SW_MENU:       // SW4(長押し)
        j_exc = J_PULSE;            // パルス出力に戻る
        break;
    }
 //  エンコーダup/down入力で数値変更
    if(f_cntud){                // エンコーダup/downあり?
        cli();                  // 割り込み禁止で読み出し
        d = cnt_add;            // エンコーダup/down値
        cnt_add  = 0;           // ゼロクリアして
        f_cntud = 0;            // 次入力待ちに
        sei();                  // 割込有効に戻す
        if((para_data >= 1000) &&       // 現在値1000以上
           (tm_encacc < (20*4))){       // 20ms未満の操作
            d *= 10;                    // 加算値を10倍してアクセル処理
        }
        tm_encacc = 0;          // アクセルチェックタイマーをクリア
        para_data += d;         // パラメータデータに±加算
        if(para_data > para_max)   para_data = para_max;      // max
        if(para_data < para_min)   para_data = para_min;      // min
        LCD.setCursor(10, menu_cp - menu_dp);    // カーソル中央
        paradstr();             // パラメータデータを文字に
        LCD.noCursor();         // カーソルなし
        LCD.print(para_str);    // 現在値表示
        j_exc--;
    }
}

/****************************/
/*      Sweep実行           */
/****************************/
//  sweep実行用データ
byte jsweep_exc;            // 実行区分
                            // 0:rise 1:wait-H 2:fall 3:wait-L
//  sweepタイマー,duty値 tm_sweepはダウンカウント
word jswp_tim;              // tm_rise,fall,wait_hi,wait_loをセット
                            // 0.1秒単位 10msごとに制御
long jswp_freq;             // 現在周波数 (0.1Hz単位)
long jswp_freq1;            // 開始周波数
long jswp_freq2;            // 停止周波数
//  タイマーとduty
struct st_jswp{
    const long  *tim   PROGMEM;  // タイマー
    const long  *freq1 PROGMEM;  // 開始周波数
    const long  *freq2 PROGMEM;  // 停止周波数
};
const st_jswp jswp_tbl[] PROGMEM={
    &sweep_tm_rise, &freq_hz[0], &freq_hz[1], // 0 Tm Rise
    &sweep_wait_hi, &freq_hz[1], &freq_hz[1], // 1 Wait Hi
    &sweep_tm_fall, &freq_hz[1], &freq_hz[0], // 2 Tm Fall
    &sweep_wait_lo, &freq_hz[0], &freq_hz[0], // 3 Wait Lo
};
//  Rise Wait-H Fall Eait-L パターン
const char msg_swp0[]  PROGMEM = "A->B  ";
const char msg_swp1[]  PROGMEM = "Wait-B";
const char msg_swp2[]  PROGMEM = "A<-B  ";
const char msg_swp3[]  PROGMEM = "Wait-A";
PGM_P msg_swp[]={
    msg_swp0,          // 0 Rise    A->B
    msg_swp1,          // 1 Wait-H  
    msg_swp2,          // 2 Fall    A<-B
    msg_swp3,          // 3 Wait-L  
};

/*****  Sweep時間表示   *****/
//  0.1s単位でtm_sweepを表示
void swpdisptm(void)
{
word d;
    cli();                      // 割り込み禁止で
    d = tm_sweep;               // 10msダウンカウントタイマー
    sei();
    d /= 10;                    // 10ms→0.1秒に
    LCD.setCursor(10, 1);       // 中央下行
    sprintf_P(str_bff, PSTR("%3u.%1u"),  // xxxx.x
            d / 10, d % 10);
    LCD.print(str_bff);         // 液晶表示
}

/*****  Sweep実行    *****/
//  表示画面をセット
void jsweep(void)
{
    LCD.clear();            // 液晶画面消去
    LCD.noBlink();          // 点滅なし
    LCD.noCursor();         // カーソルなし
    LCD.setCursor(0, 0);    // 左上
    LCD.print(F("Swp"));    // "Sweep"表示
    f_wave_disp   = 1;
    wavedisp();             // AD9833出力波形表示
    f_khz_disp    = 1;
    khzdisp();              // 周波数単位表示
    f_enccuroff   = 1;      // エンコーダ入力位置カーソル表示なし
    AD98_ch       = 0;      // 周波数出力はch Aで
//  エンコーダ入力をクリア
    cli();                  // いったん割込禁止
    f_cntud = 0;            // カウントパルスなしに
    cnt_add = 0;            // 加算値ゼロクリア
    sei();                  // 割込再開
//  表示指令
    f_swon      = 0;        // SWの再押しを待つ
//  実行開始
    jsweep_exc = 0;         // 実行区分 riseから
    j_exc++;                // next
}
/*****  Sweep実行 #1 *****/
//  jsweep_excごとの初期値をセット
void jsweep1(void)
{
long *p;
    p = pgm_read_word(&jswp_tbl[jsweep_exc].tim);   // タイマー 0～600.0s
    jswp_tim   = 10 * (word)(*p);       // タイマー値 0.1s→10m単位 max60000
    p = pgm_read_word(&jswp_tbl[jsweep_exc].freq1);  // 開始周波数
    jswp_freq1 = *p;
    p = pgm_read_word(&jswp_tbl[jsweep_exc].freq2);  // 終了周波数
    jswp_freq2 = *p;
    jswp_freq  = jswp_freq1;            // 最初の周波数
    cli();                              // 割り込み禁止で
    tm_sweep = jswp_tim;                // タイマーセット
    f_10ms   = 0;                       // 周波数更新サイクル
    f_100ms  = 0;                       // 周波数,タイマー表示タイミング
    sei();
    LCD.setCursor(4, 1);      // Rise Wait-H Fall Wait-H表示
    strcpy_P(str_bff, msg_swp[jsweep_exc]);
    LCD.print(str_bff);                 // 液晶表示
    frqdisp(jswp_freq);       // 周波数表示
    swpdisptm();              // 時間表示
//  sweepトリガ信号 Rise時にH
    if(jsweep_exc == 0)     SWPRISE_H;  // PD4 H (6pin)
    else                    SWPRISE_L;
//  next
    j_exc++;            // next
}

/*****  Sweep実行 #2 *****/
//  10msごとにduty計算
//  タイムアップで次へ rise,wait-H,fall,wait-L
//  100msサイクルでdutyと時間を表示
void jsweep2(void)
{
word t, tm;
byte *cp;   // カーソル位置
word d;
    lowbat();                   // LowBat表示
    wavedisp();                 // AD9833出力波形表示
//  表示タイミング
    if(f_100ms){                // 100msごと表示
      f_100ms = 0;
      frqdisp(jswp_freq);       // 周波数表示
      swpdisptm();              // 時間表示
//    enccuron();               // エンコーダー入力位置カーソル表示
    }
//  周波数計算
    if(f_10ms){                 // 10msごと
      f_10ms = 0;
      cli();                    // 割り込み禁止で
      t = tm_sweep;             // 10msダウンカウントタイマー
      sei();
      tm = jswp_tim - t;        // 経過時間 10ms単位
      if(jswp_tim == 0){            // タイマーがゼロならmap計算しない
        jswp_freq = jswp_freq1;     // ゼロ除算になるのでfreq1で
      }
      else{
        jswp_freq = map32(tm, 0,   jswp_tim,   // in  X1,X2 タイマー値
                     jswp_freq1, jswp_freq2);  // out Y1,Y2 デューティ
      }
      AD98freq(0, jswp_freq);                  // ch Aで周波数出力 
      if(t == 0){       // タイムアップ
        frqdisp(jswp_freq);     // 周波数表示
        swpdisptm();            // 時間表示
        j_exc = J_SWEEP3;       // 次へ rise,wait-H,fall,wait-L
      }
    }
//  スイッチ入力
    switch(swonchk()){  // スイッチ入力
      case SW_FRQ:      // SW1 出力周波数 A/B切り替え
      case SW_CURL:     // SW2 カーソル←
      case SW_CURR:     // SW3 カーソル→
      case SW_SWP1:     // SW2長押し
      case SW_SWP2:     // SW3長押し
        j_exc  = J_PULSE;       // パルス出力へ
        break;
      case SW_WAVE:     // SW4 波形選択 Sin,Tri,Sqr
        AD98_mode++;
        if(AD98_mode > M98_SQR)   AD98_mode = 0;
        AD98ctrl(0);                // 出力波形変更
        f_wave_disp   = 1;          // AD9833出力波形表示
        break;
      case SW_EEP:      // SW1長押し EEPORM書き込み
        saveeprom();            // パラメータ保存
        break;
      case SW_MENU:     // SW4長押し メニュー表示
        j_exc = J_MENU;         // メニュー表示へ
        break;
    }
}

/*****  Sweep実行 次の出力へ    *****/
//  rise,wait-H,fall,wait-L切り替え 
//  タイマー設定値がゼロのときはいきなりここへ
void jsweep3(void)
{
    jsweep_exc++;                       // 次実行区分
    if(jsweep_exc >= DIMSIZ(jswp_tbl)){ // rise～wait-L
        jsweep_exc = 0;                 // riseに戻す
    }
    j_exc = J_SWEEP1;       // 周波数,タイマー設定から
}

/************************************/
/*      メインループ実行テーブル    */
/************************************/
/*****  テーブルで実行        *****/
void (*job_tbl[])(void)={
    jttl,                   //  0   タイトル表示 
    jttl1,                  //  1
    jttl2,                  //  2
    jpulse,                 // *3   パルス出力実行
    jpulse1,                //  4
    jmenu,                  // *5   メニュー
    jmenu1,                 //  6
    jpara,                  // *7   パラメータ設定
    jpara1,                 //  8 
    jpara2,                 //  9 
    jsweep,                 // *10  Sweep実行
    jsweep1,                // *11  Sweep準備
    jsweep2,                //  12
    jsweep3,                // *13  次sweep
};

/***************************/
/*    ＳＥＴＵＰ           */
/***************************/
/*****  SET UP    *****/
void setup()
{
    cli();                  // 割込禁止
//  I/Oイニシャル
    PORTB = 0b00111100;     // data/pull up
    DDRB  = 0b00101111;     // port指定
    //          |||||+----  PB0  IO8    out -
    //          ||||+-----  PB1  IO9    out OC1A 出力  8MHzパルス出力
    //          |||+------  PB2  IO10   out AD9833 FSYNC (SS:Lアクティブ)
    //          ||+-------  PB3  IO11   out AD9833 SDATA
    //          |+--------  PB4  IO12   in  (MISO)
    //          +---------  PB5  IO13   out AD9833 SCLK ↓でSDATAラッチ
    PORTC = 0b00111111;     // data/pull up
    DDRC  = 0b00000000;     // port指定
    //          |||||+----  PC0  AD0    in  SW1入力(Lでon)
    //          ||||+-----  PC1  AD1    in  SW2
    //          |||+------  PC2  AD2    in  SW3
    //          ||+-------  PC3  AD3    in  SW4
    //          |+--------  PC4  AD4    I2C SDA(液晶)
    //          +---------  PC5  AD5    I2C SCL(液晶)
    PORTD = 0b00001111;     // data/pull up
    DDRD  = 0b01110010;     // port指定
    //        |||||||+----  PD0  IO0    in  RXD
    //        ||||||+-----  PD1  IO1    out TXD
    //        |||||+------  PD2  IO2    in  ロータリーエンコーダA相入力
    //        ||||+-------  PD3  IO3    in  ロータリーエンコーダB相入力
    //        |||+--------  PD4  IO4    out Sweepトリガ信号 Rise時にH
    //        ||+---------  PD5  IO5    out -
    //        |+----------  PD6  IO6    out -
    //        +-----------  PD7  IO7    in  コンパレータ入力(電池電圧) 
//  タイマー0 (システムで使っている)
//    そのままに
//  タイマー1 PWM出力 (OC1B 8MHz出力)
    TCCR1A = 0b11000010;    // PWM_OFF状態
    //         ||||  ++---- 高速PWM ICR1 ﾓｰﾄﾞ
    //         ||++-------- COM1B OC1B オフ (PB2)
    //         ++---------- COM1A OC1A 8MHz出力 (PB1)
    TCCR1B = 0b00011001;
    //         || ||+++---- クロック選択 16MHz
    //         || ++------- 高速PWM ICR1 ﾓｰﾄﾞ
    //         |+---------- ICES1
    //         +----------- ICNC1
    ICR1   = 2 - 1;         // 1/2 (TOP値)
    OCR1A  = 0;             // PWM duty 50%
    TIMSK1 = 0b00000000;    // 割り込み 使わない
    //           |  ||+---  TOIE1
    //           |  |+----  OCIE1A
    //           |  +-----  OCIE1B
    //           +--------  ICIE1
//  タイマー2  4kHz割り込み
    TCCR2A = 0b00000010;
    //         ||||  ++---- WGM21,20 OCR2A コンペアマッチ
    //         ||++-------- COM2B
    //         ++---------- COM2A
    TCCR2B = 0b00000011;
    //             |+++---  CLK/32 = 16MHz / 32 = 500kHz 2us
    //             +------  WGM02
    OCR2A  = 125  - 1;       // 500kHz/125  = 4kHz 0.25ms
    TIMSK2 = 0b00000010;    // 割り込み
    //              ||+---  TOIE2
    //              |+----  OCIE2A 割込on
    //              +-----  OCIE2B
//  コンパレータ LowBat検出
    ACSR   = 0b01000000;     // コンパレータ設定
    //         ||||||++---  ACIS 割り込み条件
    //         |||||+-----  ACIC 割り込みは使わない
    //         ||||+------  ACIE
    //         |||+-------  ACI
    //         ||+--------  ACO  HならLowBat
    //         |+---------  ACBG 1.1V基準電圧使用
    //         +----------  ACD  コンパレータ使う
    ADMUX  = 0b11001111;     // VREF設定    
    //         ||| ++++---  MUX3～0 GNDに
    //         ||+--------  ADLAR
    //         ++---------  REFS    内部1.1Vに AREFピンにコンデンサ可
    DIDR1  = 0b00000010;     // デジタル入力禁止
    //               |+---  AIN0D
    //               +----  AIN1D PD7 LowBat検出
//  シリアル,液晶:I2C
    sei();                  // 割込許可
    Serial.begin(9600);     // シリアル 9.6kBPSで 出力だけ使う
//  シリアル受信禁止
    cli();                  // 割込禁止
    UCSR0B &= 0b01101111;   // 受信,受信割込禁止
    //          |||||||+--  TXB8
    //          ||||||+---  RXB8
    //          |||||+----  UCSZ2
    //          ||||+-----  TXEN
    //          |||+------  RXEN  ←受信なし
    //          ||+-------  UDRIE
    //          |+--------  TXCIE
    //          +---------  RXCIE ←受信割込禁止
    sei();                  // 割込許可
//  AD9833とのインターフェース SPI初期化
//  MOSIとCLK,SSはHの状態で
    SPCR    = 0b01011000;   // 受信,受信割込禁止
    //          ||||||++--  SPR  クロック 16MHz/4
    //          ||||++----  CPOL,CPHA  CLK Hで↓エッジ
    //          |||+------  MSTR  マスターに
    //          ||+-------  DORD  MSBから転送
    //          |+--------  SPE   SPIイネーブル
    //          +---------  SPIE  割り込みは無し
//  液晶 LCD_V5R0,LCD_V3R3で電源電圧を区分 ★
    LCD.begin(LCD_COLS, LCD_ROWS, LCD_V5R0);     // LCD初期化
                        // 文字数、行数設定 (16文字x2行) 5V電源
    lcdcgset();         // LCD キャラジェネ設定
//  AD9833リセット
    AD98reset();          // リセットして出力の準備
}

/******************************/
/*      ループ                */
/******************************/
/*****  LOOP      *****/
void loop()
{
    while(1){
        job_tbl[j_exc]();       // メイン処理実行
    }
};
/*====  end of "AD9833_01.ino"   ====*/
