/*****  AQM1602Y-FLW-FBW I2C 液晶表示ライブラリ *****/
//  「LiquidCrystal」を元に改造
//  2022-07-19  setContrast,getContrast  LCD_V5R0,LCD_V3R3に
//  2024-04-20  ライブラリ、wire,twiを使わず割り込みを用いない方法で
#include "i2c_lcd_aqm1602a.h"
#include <Arduino.h>
/********************************/
/*  ＴＷＩモジュールのアクセス  */
/********************************/
//  ATmega328PのTWIハードウェアを制御
//  TWINT=1待ちに割り込みを使わなない
//  タイムアウトチェックカウンタを設けて異常でも抜けるように
/*****  TWI 初期化      *****/
//  液晶モジュールアドレスをセット
//  TWI周波数を設定 100kHz→200kHz
void twiInit(void )
{
//  ATmega328P設定
    TWSR &= 0b11111100;     // プリスケーラ 1/1に
        //          ++----- TWPS1,0
    TWBR  = 32;             // クロック200kHzに
                            // SCL = 16MHz / (16+(2*TWBR))
    TWCR  = 0b00000100;
        //    |||||| +----- TWIE  割込は使わない
        //    |||||+------- TWEN  TWI動作許可
        //    ||||+-------- TWWC
        //    |||+--------- TWSTO
        //    ||+---------- TWSTA
        //    |+----------- TWEA  ACK応答許可
        //    +------------ TWINT
}

/*****  TWI 書込み送出開始    *****/
void twiStartW(uint8_t adrs)
{
uint16_t tt;        // time out check counter
    TWCR  = 0b10100100;
        //    |||||| +----- TWIE  割込は使わない
        //    |||||+------- TWEN  TWI動作許可
        //    ||||+-------- TWWC
        //    |||+--------- TWSTO
        //    ||+---------- TWSTA 開始
        //    |+----------- TWEA
        //    +------------ TWINT TWINTクリア
    tt = 2000;      // 2000 loop
    while(tt){      // time out?
      if((TWCR & (1 << TWINT)))   break;  // TWINT=1を待つ
      tt--;
    }
    TWDR = adrs << 1;       // アドレス+W(0)
    TWCR  = 0b10000100;
        //    |||||| +----- TWIE  割込は使わない
        //    |||||+------- TWEN  TWI動作許可
        //    ||||+-------- TWWC
        //    |||+--------- TWSTO
        //    ||+---------- TWSTA
        //    |+----------- TWEA
        //    +------------ TWINT TWINTクリア
    tt = 2000;      // 2000 loop
    while(tt){      // time out?
      if((TWCR & (1 << TWINT)))   break;  // TWINT=1を待つ
      tt--;
    }
}

/*****  TWI データ書き込み  *****/
void twiWrite(uint8_t d)
{
uint16_t tt;        // time out check counter
    TWDR = d;
    TWCR  = 0b10000100;
        //    |||||| +----- TWIE  割込は使わない
        //    |||||+------- TWEN  TWI動作許可
        //    ||||+-------- TWWC
        //    |||+--------- TWSTO
        //    ||+---------- TWSTA
        //    |+----------- TWEA
        //    +------------ TWINT TWINTクリア
    tt = 2000;      // 2000 loop
    while(tt){      // time out?
      if((TWCR & (1 << TWINT)))   break;  // TWINT=1を待つ
      tt--;
    }
}

/*****  TWI 送出終了    *****/
void twiStop(void)
{
    TWCR  = 0b10010100;
        //    |||||| +----- TWIE  割込は使わない
        //    |||||+------- TWEN  TWI動作許可
        //    ||||+-------- TWWC
        //    |||+--------- TWSTO 終了
        //    ||+---------- TWSTA
        //    |+----------- TWEA
        //    +------------ TWINT TWINTクリア
}

/*********************/
/*   液晶アクセス    */
/*********************/

/*****  コンストラクタ：I2Cアドレス設定     *****/
i2c_lcd_aqm1602::i2c_lcd_aqm1602(uint8_t adrs) {
    _i2c_adrs = adrs;               // I2Cアドレス
}

/*****  文字数、行数設定    *****/
//  液晶のイニシャル処理 　文字数と行数を設定
//  I2Cの開始「twibegin();」はメインのsetup()で実行する
//  v:電源電圧区分 0なら5V、1なら3.3V
void i2c_lcd_aqm1602::begin(uint8_t cols, uint8_t lines, uint8_t v) {
uint8_t i;
    delayMicroseconds(20000);   // 20ms    
    twiInit();                  // I2C開始
    delayMicroseconds(10000);   // 10ms    
    _numcols  = cols;               // 設定した文字数を保存
    _numlines = lines;              // 行数保存
    _row_offsets[0]  = 0x00;        // 1行目　表示アドレス設定
    _row_offsets[1]  = 0x40;        // 2行目　行間は連続していない
    _row_offsets[2]  = 0x00 + cols; // 3行目　+文字数を
    _row_offsets[3]  = 0x40 + cols; // 4行目
//  コマンドデータ
    _displayfunction = 0b00111000;  // LCD機能設定 <0x38>
                //         |||| +--  0:IS 機能選択
                //         |||+----  0:8行フォント 1:11行フォント
                //         ||+-----  0:1行表示     1:2行表示 
                //         |+------  0:4bitモード  1:8bitモード
                //         +-------  0x20 コマンド
    _displaycontrol  = 0b00001100;  // 表示,カーソルon/off (0x0C>
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    _displaymode     = 0b00000110;  // エントリーモード  <0x06>  
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    if(v == LCD_V5R0){              // 電源5Vの時
      _disppower     = 0b01010000;  // Bon = 0 <0x50>
      _dispfollower  = 0b01101100;  // Foollower Fon Rbx2.0
    }
    else{                           // 電源3.3V
      _disppower     = 0b01010100;  // Bon = 1 <0x54>
      _dispfollower  = 0b01101100;  // Foollower Fon Rbx2.0
    }
    _contrast = 35;                 // コントラスト(6bit) 0～63 暫定値 ★
//  イニシャルコマンド送出 (昔は0x38を3回送っていたけれど・・・)
    command(_displayfunction);              // 機能設定 <0x38>
    command(_displayfunction | 0b00000001); // <0x39> IS=1
    command(0b00010100);            // 周波数  <0x14>
    //            |+++----- F2,1,0 192kHz(5V) 183kHz(3.3V)
    //            +-------- BS 0:bias 1/5
    command(LCD_CONTRAST | (_contrast & 0x0F));      // コントラスト下位 <0x70>
    command(_disppower | ((_contrast >> 4) & 0x03)); // コントラスト上位 <0x50>
    command(_dispfollower);         // Follower <0x6C>
    for(i = 0; i < 200; i++){       // 200ms wait
        delayMicroseconds(1000);    // 1000us
    }
    command(_displayfunction);      // 機能設定 <0x38> IS=0
    command(_displaycontrol);       // 表示 on  <0x0C>
    command(_displaymode);          // エントリーモード <0x06>
    clear();                        // 画面消去,DDRAM=0
}

/*****  表示クリア      *****/
void i2c_lcd_aqm1602::clear() {
    command(LCD_CLEARDISPLAY);  // clear display, set cursor position to zero
    delayMicroseconds(2000);    // this command takes a long time!
}

/*****  カーソル原点位置へ  *****/
void i2c_lcd_aqm1602::home() {
    command(LCD_RETURNHOME);    // set cursor position to zero
    delayMicroseconds(2000);    // this command takes a long time!
}

/*****  カーソル移動    *****/
//  文字コラム位置(0～cols-1) ,行(0～lines-1)
//  行位置は規制, コラム位置は規制なしで
void i2c_lcd_aqm1602::setCursor(uint8_t col, uint8_t row) {
    if ( row >= _numlines ) {     // 文字行位置
         row  = _numlines - 1;    // 最下行で
    }
    command(LCD_SETDDRAMADDR | (col + _row_offsets[row]));  // DDRAMアドレス
}

/*****  表示 on/off *****/
void i2c_lcd_aqm1602::noDisplay() {
    _displaycontrol &= 0b11111011;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602::display() {
    _displaycontrol |= 0b00000100;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  カーソル on/off *****/
void i2c_lcd_aqm1602::noCursor() {
    _displaycontrol &= 0b11111101;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602::cursor() {
    _displaycontrol |= 0b00000010;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  カーソル点滅 on/off *****/
void i2c_lcd_aqm1602::noBlink() {
    _displaycontrol &= 0b11111110;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}
void i2c_lcd_aqm1602::blink() {
    _displaycontrol |= 0b00000001;  // 表示,カーソルon/off
                //           |||+--  0:文字点滅off 1:文字点滅on
                //           ||+---  0:カーソルoff 1:カーソルon
                //           |+----  0:表示off     1:表示on
                //           +-----  0x08 コマンド
    command(_displaycontrol);
}

/*****  画面スクロール  *****/
void i2c_lcd_aqm1602::scrollDisplayLeft(void) {
    command(0b00011000);        // カーソル移動、表示シフト
    //           ||+------  0:左シフト 1:右シフト
    //           |+-------  0:表示シフトしない 1:表示シフトする
    //           +--------  0x10 コマンド
}
void i2c_lcd_aqm1602::scrollDisplayRight(void) {
    command(0b00011100);        // カーソル移動、表示シフト
    //           ||+------  0:左シフト 1:右シフト
    //           |+-------  0:表示シフトしない 1:表示シフトする
    //           +--------  0x10 コマンド
}

/*****  入力方向切替    *****/  
void i2c_lcd_aqm1602::leftToRight(void) {
    _displaymode    |= 0b00000010;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}
void i2c_lcd_aqm1602::rightToLeft(void) {
    _displaymode    &= 0b11111101;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}

/*****  スクロール(画面シフト) on/off   *****/
void i2c_lcd_aqm1602::autoscroll(void) {
    _displaymode    |= 0b00000001;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}
void i2c_lcd_aqm1602::noAutoscroll(void) {
    _displaymode    &= 0b11111110;  // エントリーモード    
                //            ||+--  0:表示シフトなし 1:表示シフトあり
                //            |+---  0:デクリメント   1:インクリメント 
                //            +----  0x04 コマンド
    command(_displaymode);          // エントリーモード
}

/*****  コントラスト設定 ★   *****/
//  n : 0～63 6bit
void i2c_lcd_aqm1602::setContrast(uint8_t n) {
    if(n > 63)          n = 63;     // 0～63
    _contrast = n;
    command(_displayfunction | 0b00000001); // <0x39> IS=1
    command(LCD_CONTRAST |  (_contrast & 0x0F));       // コントラスト下位
    command(_disppower   | ((_contrast >> 4) & 0x03)); // コントラスト上位
    command(_displayfunction);              // <0x38> IS=0
}
/*****  コントラスト設定値読み出し ★   *****/
//  ret : 0～63 6bit
uint8_t i2c_lcd_aqm1602::getContrast() {
    return _contrast;           // 0～63 現在の設定値
}

/*****  CG RAM 設定     *****/
//  location:0～7,  charmap:8x8bitのキャラジェネデータ
void i2c_lcd_aqm1602::createChar(uint8_t location, uint8_t charmap[]) {
    location &= 0x7;                // we only have 8 locations 0-7
    command(LCD_SETCGRAMADDR | (location << 3));
    for (byte i = 0; i < 8; i++) {
        write(charmap[i]);      // CGデータを転送
    }
}

/*****  コマンド送出    *****/
void i2c_lcd_aqm1602::command(uint8_t value) {
    twiStartW(_i2c_adrs);
    twiWrite(0x00);                   // RS = 0
    twiWrite(value);
    twiStop();
    delayMicroseconds(40);            // 40us wait
}

/*****  データ送出  *****/
size_t i2c_lcd_aqm1602::write(uint8_t value) {
    twiStartW(_i2c_adrs);
    twiWrite(0x40);                   // RS = 1
    twiWrite(value);
    twiStop();
    delayMicroseconds(40);            // 40us wait
    return 1;                         // assume sucess
}

// end of "i2c_lcd_aqm1602.cpp"
