/*******************************************************/
/*    HOCOUTRを操作したときのHCO周波数の変化をチェック  */
/*******************************************************/
//  HOCO_02.ino   2025-05-15    JH3DBO
//  入出力ピン
//      D2   P105  GPT1A  キャプチャ入力 外部から基準の1Hzを入力
//      D3   P104  GPT1B  クロック入力 測定する1MHzを入力
//      D4   P103  GPT2A  1MHzパルス出力 これを測定
//      D8   P304  GPT7A  テスト用に1Hzパルスを出力
//  データはSerial1に出力 (USBには出力しない)
//  タイミングチェック用出力
//      D9   P303  loopでH/Lパルス
//      D10  P112  1秒カウント中Hに
//      D13  P111  LEDポート　100msでトグル
//  ポートH/L制御出力
#define D9_H    (R_PORT3->PODR_b.PODR3  = 1)    // D9:P303 H/L
#define D9_L    (R_PORT3->PODR_b.PODR3  = 0)        
#define D10_H   (R_PORT1->PODR_b.PODR12 = 1)    // D10:P112 H/L
#define D10_L   (R_PORT1->PODR_b.PODR12 = 0)        
#define D13_X   (R_PORT1->PODR_b.PODR11 = !(R_PORT1->PODR_b.PODR11)) // D13トグル       
//  タイマー1インプットキャプチャフラグ
#define CK_TCFA  (R_GPT1->GTST_b.TCFA)          // GPT1インプットキャプチャチェック
#define CLR_TCFA (R_GPT1->GTST_b.TCFA = 0)      // インプットキャプチャクリア
//  文字出力バッファ
char str_bff[40];           // sprintfで使用
/*****  SＥＴＵＰ   *****/
void setup(){
//  ポート設定
    pinMode(9,  OUTPUT);        // D9:P303  チェックパルス用ポート
    pinMode(10, OUTPUT);        // D10:P112
    pinMode(13, OUTPUT);        // D13:P111 LED
//  モジュールストップ解除
    R_MSTP->MSTPCRD_b.MSTPD5  = 0;  // 32bit PWMモジュールストップ解除
    R_MSTP->MSTPCRD_b.MSTPD6  = 0;  // 16bit PWMモジュールストップ解除
    R_MSTP->MSTPCRD_b.MSTPD14 = 0;  // POEGモジュールストップ解除
    R_SYSTEM->PRCR  = 0xA501;       // システムプロテクト解除
                                    // 解除のまま
//  D2:P105  GPT1A  キャプチャ入力 外部から基準の1Hzを入力
    R_PFS->PORT[1].PIN[5].PmnPFS_b.PCR  = 1;        // P105入力pullup
    R_PFS->PORT[1].PIN[5].PmnPFS_b.PSEL = 0b00010;  // GTETRGA入力に
    R_PFS->PORT[1].PIN[5].PmnPFS_b.PMR  = 1;        // 周辺機能有効
    R_GPT1->GTICASR_b.ASGTRGAF          = 1;        // A↓でキャプチャ
//  D3:P104  GPT1B  クロック入力 upカウント
    R_PFS->PORT[1].PIN[4].PmnPFS_b.PCR  = 1;        // P105入力pullup
    R_PFS->PORT[1].PIN[4].PmnPFS_b.PSEL = 0b00010;  // GTETRGB入力に
    R_PFS->PORT[1].PIN[4].PmnPFS_b.PMR  = 1;        // 周辺機能有効
    R_GPT1->GTUPSR_b.USGTRGBF           = 1;        // B↓でカウントアップ
    R_GPT1->GTCR_b.CST                  = 1;        // カウント開始
//  D4:P103 GPT2A 出力にして1MHzパルス出力
    R_PFS->PORT[1].PIN[3].PmnPFS_b.PDR  = 1;        // P103ポート出力に
    R_PFS->PORT[1].PIN[3].PmnPFS_b.PSEL = 0b00011;  // GTIOC2A
    R_PFS->PORT[1].PIN[3].PmnPFS_b.PMR  = 1;        // 機能出力有効
    R_GPT2->GTCR_b.MD   = 0b000;        // ノコギリ波モード
    R_GPT2->GTCR_b.TPCS = 0b000;        // PCLKD/1 48MHz
    R_GPT2->GTPR        = 24 - 1;       // 48MHz/24=2MHz トグルで1/2して1MHz
    R_GPT2->GTCNT       = 0;            // 初回カウント値
    R_GPT2->GTIOR_b.GTIOA = 0b01100;    // GTIOC2A 周期の終わりでトグル出力
    R_GPT2->GTIOR_b.OAE   = 1;          // A出力許可
    R_GPT2->GTCR_b.CST    = 1;          // カウント開始
//  D8:P304 GPT7A出力 1Hzパルス発生 
    R_PFS->PORT[3].PIN[4].PmnPFS_b.PDR  = 1;        // P302ポート出力
    R_PFS->PORT[3].PIN[4].PmnPFS_b.PSEL = 0b00011;  // GTIOC7A出力に
    R_PFS->PORT[3].PIN[4].PmnPFS_b.PMR  = 1;        // 周辺機能有効
    R_GPT7->GTCR_b.TPCS = 0b1010;       // PCLKD/1024 = 46875Hz
    R_GPT7->GTPR        = 46875 - 1;    // 1Hz
    R_GPT7->GTCCR[0]    = 46875 / 2;    // GTIOC7AがHになるカウント
    R_GPT7->GTIOR_b.GTIOA = 0b00110;    // GTIOC7A 周期の終わりでL, A一致でH
    R_GPT7->GTIOR_b.OAE   = 1;          // A出力許可
    R_GPT7->GTCR_b.CST    = 1;          // カウント開始
}
/*****  ＬＯＯＰ    *****/
void loop()
{
byte  f_cap0 = 0;           // 最初のキャプチャ
uint32_t cnt_0, cnt_1;      // GPT1(32bit)キャプチャ前回値と今回値
uint32_t cnt_frq;           // 1秒で周波数(Hz)データ
int   ut_r;                 // HOCO User Trimレジスター設定値
                            // -10～0～+10と変化させて1MHz出力の周波数変化を見る
uint32_t ms , a;            // 1msチェックデータ
word  tm_1ms = 0;           // 1msダウンカウントタイマー
int   m_exc  = 0;           // 計測実行区分 switchで1秒ごとに処理
int   m_nbr  = 0;           // 測定番号
//  Serial1でデータ出力
    Serial1.begin(9600);      // 9600BPSで
    Serial1.println("# Test HOCOUTCR (HOCO_02)");
    ms = millis();            // 現1ms値
//  loop
    while(1){
      D9_H;                   // D9:P303 H/L
      D9_L;
      a = millis();               // 新1ms値
      if(ms != a){                // 1ms変化?
        ms = a;                   // 次の1msを待つ
        if(tm_1ms)    tm_1ms--;   // 1msダウンカウント
        if(tm_1ms == 0){          // タイムアップ?
          D13_X;                  // LEDをトグル
          tm_1ms = 100;           // 100msをセット
        }
      }
//  キャプチャをチェック
      if(CK_TCFA != 0){       // キャプチャ発生
          CLR_TCFA;           // フラグクリア
          cnt_1 = R_GPT1->GTCCR[0];   // CAP Aデータ
          if(f_cap0 == 0){            // 初めてのキャプチャ
              cnt_0 = cnt_1;          // cap0にコピー
              f_cap0 = 1;
          }
          cnt_frq = cnt_1 - cnt_0;    // 前回値との差分
          cnt_0 = cnt_1;              // cap0にコピー
//  m_excで処理を変える
          switch(m_exc){
            case 0:   // ut_rを-10にセット
              ut_r  = -10;                // HOCO Trimレジスタ -10から
              R_SYSTEM->HOCOUTCR = ut_r;  // HOCOデータ更新
              sprintf(str_bff, "# %d", m_nbr);   // 測定番号
              Serial1.println(str_bff);
              m_nbr++;                    // 番号+1
              m_exc++;                    // 次stepへ
              break;
            case 1:   // 1秒待ち  (ここでloop)
              D10_H;                      // D10:P112 H
              m_exc++;
              break;
            case 2:   // 周波数データ読み取り
              D10_L;                      // D10:P112 L
              sprintf(str_bff, "%3d %3d% 6ld.%03ldkHz",
                        ut_r,                             // 書き込み値
                        (int)(int8_t)R_SYSTEM->HOCOUTCR,  // 念のため読んでみる
                        cnt_frq / 1000,       // 周波数データ(kHzで)
                        cnt_frq % 1000);
              Serial1.println(str_bff);
              ut_r++;                   // トリムデータを+1
              if(ut_r > 10){            // -10～0～10
                Serial1.println();      // 改行 2つ
                Serial1.println();
                m_exc = 0;              // -10から再スタート
              }
              else{
                R_SYSTEM->HOCOUTCR = ut_r;  // HOCOデータ更新
                m_exc = 1;                  // 1秒待ちへ
              }
              break;
          }          
      }
    }
}
