/*
 * =====================================================================
 *  E12系列表示対応コンデンサ容量計
 * =====================================================================
 * 概要：
 *   - 測定値に最も近いE12系列標準値を自動判定し、差分・誤差率を表示
 *   - RC時定数による測定（実用範囲: 1nF～5μF程度）
 *   - 大容量コンデンサ向けタイムアウト推定計算機能
 *   - ADC値間の線形補間による測定時刻精度向上
 *   - シリアルモニタ＋I2C LCD(16x2)への同時出力
 *   - 長時間測定時の進捗表示機能
 *   - 起動時端子開放チェックによる安全なキャリブレーション
 *   - 測定不能時の自動待機機能
 * 
 * 測定原理：
 *   1MΩ抵抗によるRC充電回路、63.2%充電時間から容量算出
 *
  * © JH4VAJ 2025
 * =====================================================================
 */

// === ライブラリ ===
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <math.h>

// === 動作設定 ===
//#define DEBUG                    // デバッグ表示を有効にする

// === 測定パラメータ ===
const unsigned long CHARGE_TIMEOUT_US = 5000000UL; // 充電タイムアウト（μs）5秒
const int MIN_ADC_FOR_MEASUREMENT   = 200;       // タイムアウト時の測定可能最小ADC値
const int ADC_SAMPLE_HOLD_DELAY_US  = 12;        // ADCサンプル&ホールド遅延補正（μs）
                                                 // Arduino ADCクロック125kHzでの1.5クロック分（1.5/125kHz = 12μs）
const int DEFAULT_DISCHARGE_TIME    = 100;       // 放電時間（ms）
                                                 // 【例】放電用抵抗220Ω・コンデンサ10μFの場合、99.9%放電にかかる時間は約15.2ms
const unsigned long MIN_CHARGE_TIME_US = 200UL;  // コンデンサ未接続判定用

// === ハードウェア設定 ===
const int PULSE_PIN = 2;                         // 充電用デジタルピン
const int DISCHARGE_PIN = 3;                     // 放電用デジタルピン（推奨：220Ω抵抗を介してコンデンサに接続）
const int LED_PIN = LED_BUILTIN;                 // LED点滅用ピン
const int ANALOG_PIN = A1;                       // 測定ピン

// === 回路定数 ===
const float CHARGE_RESISTOR = 1000000.0f;          // 充電抵抗値（Ω）※実測値に合わせて調整すること（例：1MΩ = 1,000,000Ω）
const int ADC_DIVISIONS = 1024;                    // ADC分解能
const int TARGET_ADC = (int)(ADC_DIVISIONS * 0.632); // 基準値の63.2%（約647）

// === E12系列定義 ===
const float E12_SERIES[] = {1.0, 1.2, 1.5, 1.8, 2.2, 2.7, 3.3, 3.9, 4.7, 5.6, 6.8, 8.2};
const int E12_SIZE = sizeof(E12_SERIES) / sizeof(E12_SERIES[0]);

// === 単位変換定義 ===
const char* SI_PREFIX[] = {"", "m", "u", "n", "p"};
const float UNIT_FACTORS[] = {1e-6, 1e-3, 1.0, 1e3, 1e6};

// === LCDオブジェクト ===
LiquidCrystal_I2C lcd(0x27, 16, 2);

// === グローバル変数 ===
unsigned long offsetTime = 0;         // 浮遊容量補正値（μs）← 実測値で上書きする
unsigned long analogReadTimeUs = 100; // analogRead()の平均実測時間（μs）← 実測値で上書きする

enum ScreenState { NO_CAPACITOR, MEASUREMENT_RESULT };
ScreenState currentScreenState = NO_CAPACITOR;

/**
 * @struct ChargeResult
 * @brief measureChargeTime()による測定結果を格納する構造体
 */
struct ChargeResult {
  unsigned long elapsedTime;
  int finalADC;
};

// =====================================================================
//  ヘルパー関数群
// =====================================================================
/**
 * @brief 桁区切り付きでunsigned long値をシリアルに出力する関数
 * @param num 出力する数値
 */
void printWithComma(unsigned long num) {
  char buf[20];
  int len = 0;
  int count = 0;

  if (num == 0) {
    Serial.print("0");
    return;
  }

  // 逆順に3桁ごとにカンマ挿入
  while (num > 0) {
    if (count && count % 3 == 0) {
      buf[len++] = ',';
    }
    buf[len++] = '0' + (num % 10);
    num /= 10;
    count++;
  }

  // buf内は逆順なので反転して正順で出力
  for (int i = len - 1; i >= 0; i--) {
    Serial.print(buf[i]);
  }
}

/**
 * @brief 数値を、指定した最大文字数（符号・小数点含む）で丸めて符号付きの文字列に変換
 * @param buf 出力バッファ（最大文字数+1バイト以上必要）
 * @param value 変換対象の数値
 * @param maxChars 最大文字数（符号・小数点含む）
 */
void toFixedWidthNumberString(char* buf, float value, int maxChars) {
  char sign = (value >= 0) ? '+' : '-';
  float abs_value = fabs(value);

  // 残り文字数（符号1文字分を引く）
  int remain = maxChars - 1;
  // 最大整数部桁数
  int intDigits = (abs_value < 1) ? 1 : (int)floor(log10(abs_value)) + 1;

  // 小数点は必ず1文字使う
  int decimals = remain - intDigits - 1;
  if (decimals < 0) decimals = 0;

  // 四捨五入
  float scale = pow(10, decimals);
  float rounded = round(abs_value * scale) / scale;

  // 桁上がりで整数部が増えた場合の再調整
  int newIntDigits = (rounded < 1) ? 1 : (int)floor(log10(rounded)) + 1;
  if (newIntDigits > intDigits) {
    intDigits = newIntDigits;
    decimals = remain - intDigits - 1;
    if (decimals < 0) decimals = 0;
    scale = pow(10, decimals);
    rounded = round(abs_value * scale) / scale;
  }

  // dtostrf()を使用して浮動小数点数を文字列に変換
  char temp[16];
  dtostrf(rounded, intDigits + decimals + 1, decimals, temp);
  
  // 符号を先頭に付加
  buf[0] = sign;
  strcpy(buf + 1, temp);
  
  // 小数点以下が0桁の場合でも小数点を残す
  if (decimals == 0) {
    if (!strchr(buf, '.')) {
      int len = strlen(buf);
      buf[len] = '.';
      buf[len + 1] = '\0';
    }
  }

  // 必要なら右側をゼロパディング
  int len = strlen(buf);
  if (len < maxChars) {
    for (int i = len; i < maxChars; ++i) buf[i] = '0';
    buf[maxChars] = '\0';
  }

  // ちょうどmaxChars文字だけに調整
  buf[maxChars] = '\0';
}

/**
 * @brief 測定値をLCD用に「整数部最大3桁（左空白パディング）、
 * 小数点以下2桁、小数点位置固定、単位付き」で整形
 * @param buf 出力バッファ (9バイト以上必要)
 * @param value_uF 容量値(uF)
 * @param unit_idx 単位インデックス
 */
void formatCapacitanceForLCD(char* buf, float value_uF, int unit_idx) {
  float value = value_uF * UNIT_FACTORS[unit_idx];

  // 100倍して四捨五入
  long rounded = (long)(value * 100.0f + (value >= 0 ? 0.5f : -0.5f));
  int int_part = (int)(rounded / 100);
  int frac_part = abs((int)(rounded % 100));

  sprintf(buf, "%3d.%02d%sF", int_part, frac_part, SI_PREFIX[unit_idx]);
}

/**
 * @brief E12値をLCD用に「整数部最大3桁（左空白パディング）、
 * 小数点以下1桁、小数点位置固定、（単位はなし）」で整形
 * @param buf 出力バッファ (6バイト以上必要)
 * @param value_uF 容量値(uF)
 * @param unit_idx 単位インデックス
 */
void formatE12ValueForLCD(char* buf, float value_uF, int unit_idx) {
  dtostrf((value_uF * UNIT_FACTORS[unit_idx]), 5, 1, buf);
}

/**
 * @brief 差分値をLCD用に「符号1文字 + 数値部分（小数点以下2桁固定）」で整形
 * @param buf 出力バッファ (8バイト以上必要)
 * @param diff_uF 差分値(uF)
 * @param unit_idx 単位インデックス
 */
void formatDifferenceForLCD(char* buf, float diff_uF, int unit_idx) {
  float disp = diff_uF * UNIT_FACTORS[unit_idx];
  toFixedWidthNumberString(buf, disp, 6);
}

/**
 * @brief 誤差率をLCD用に「符号1文字 + 数値4文字 + %記号1文字」で整形
 * @param buf 出力バッファ (7バイト以上必要)
 * @param percent 誤差率
 */
void formatPercentForLCD(char* buf, float percent) {
  if (percent > 99.95f) {
    percent = 99.95f;
  } else if (percent < -99.95f) {
    percent = -99.95f;
  }
  toFixedWidthNumberString(buf, percent, 5);
  buf[5] = '%';
  buf[6] = '\0';
}

/**
 * @brief コンデンサを放電する
 */
void discharge(unsigned long discharge_time_ms) {
  pinMode(DISCHARGE_PIN, OUTPUT);
  digitalWrite(DISCHARGE_PIN, LOW);
  delay(discharge_time_ms);
  pinMode(DISCHARGE_PIN, INPUT);   // 放電完了後、ピンをハイインピーダンス状態にする
}

/**
 * @brief 充電を開始する、その時の時間を返す
 */
unsigned long startCharge() {
  digitalWrite(PULSE_PIN, HIGH);
  return micros();
}

/**
 * @brief 充電を停止する
 */
void stopCharge() {
  digitalWrite(PULSE_PIN, LOW);
}

// =====================================================================
//  測定コア関数群
// =====================================================================
/**
 * @brief analogRead()の平均処理時間（μs）を測定する関数
 * 10回連続で測定して平均値を算出
 * @return analogRead()の平均処理時間（μs）
 */
unsigned long measureAnalogReadTime() {
  analogRead(ANALOG_PIN);   // 安定化のため空読み（念のため、2回）
  analogRead(ANALOG_PIN);

  unsigned long t1 = micros();

  // ループの影響を避けるためベタ書きで10回実行
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);

  unsigned long t2 = micros();

  return (t2 - t1) / 10;
}

/**
 * @brief 長時間測定開始時の進捗表示初期化
 */
void initializeLongMeasurementDisplay() {
  if (currentScreenState == NO_CAPACITOR) {
    lcd.clear();
    lcd.setCursor(0, 0);
    lcd.print("Measuring...    ");
  }
}

/**
 * @brief 進捗表示の更新
 * @param elapsedSec 経過秒数
 */
void updateProgressDisplay(unsigned int elapsedSec) {
  if (currentScreenState == NO_CAPACITOR) {
    char line[17];
    sprintf(line, "%16u", elapsedSec);
    lcd.setCursor(0, 1);
    lcd.print(line);
  } else {
    char timeStr[4];
    sprintf(timeStr, "%3u", elapsedSec);
    lcd.setCursor(6, 1);
    lcd.print(timeStr);
  }
}

/**
 * @brief 長時間測定完了時の後処理
 */
void finalizeLongMeasurementDisplay() {
  if (currentScreenState == NO_CAPACITOR) {
    lcd.clear();
  } else {
    lcd.setCursor(6, 1);
    lcd.print("E12");
  }
}

/**
 * @brief 充電時間測定関数
 */
ChargeResult measureChargeTime() {
  ChargeResult result = {0, 0};
  
  // 放電処理
#ifdef DEBUG
  Serial.print("ADC before discharge: ");
  Serial.print(analogRead(ANALOG_PIN));
#endif

  discharge(DEFAULT_DISCHARGE_TIME);
  
#ifdef DEBUG
  Serial.print("  ADC after discharge: ");
  Serial.println(analogRead(ANALOG_PIN));
#endif

  unsigned long startTime = startCharge();
  unsigned long prevTime = startTime;
  int prevAdcValue = 0;
  unsigned int prevSec = 0;
  bool isLongMeasurement = false;

  while (true) {
    unsigned long currentTime = micros();
    int adcValue = analogRead(ANALOG_PIN);

    if (adcValue >= TARGET_ADC) {
      // 線形補間でより正確な時刻を推定
      float ratio = (float)(TARGET_ADC - prevAdcValue) / (float)(adcValue - prevAdcValue);
      unsigned long interpolatedTime = prevTime + (unsigned long)((currentTime - prevTime) * ratio);

#ifdef DEBUG
      Serial.print("Linear interpolation - prevADC: ");
      Serial.print(prevAdcValue);
      Serial.print(", currentADC: ");
      Serial.print(adcValue);
      Serial.print(", timeDiff: ");
      Serial.print(currentTime - prevTime);
      Serial.print(" us, ratio: ");
      Serial.print(ratio, 6);
      Serial.print(", timeOffset: ");
      Serial.print((unsigned long)((currentTime - prevTime) * ratio));
      Serial.println(" us");
#endif

      result.elapsedTime = interpolatedTime - startTime;
      result.elapsedTime += ADC_SAMPLE_HOLD_DELAY_US;
      result.finalADC = TARGET_ADC;
      break;
    }
    
    if (currentTime - startTime > CHARGE_TIMEOUT_US) {
      result.elapsedTime = 0;
      result.finalADC = adcValue;
      break;
    }

    unsigned int elapsedSec = (currentTime - startTime) / 1000000UL;
    if (elapsedSec > 0 && elapsedSec != prevSec) {
      prevSec = elapsedSec;

#ifdef DEBUG
      Serial.print("elapsedSec: ");
      Serial.print(elapsedSec);
      Serial.print("  adcNow: ");
      Serial.println(adcValue);
#endif

      if (!isLongMeasurement) {
        isLongMeasurement = true;
        initializeLongMeasurementDisplay();
      }

      updateProgressDisplay(elapsedSec);
    }
    
    prevTime = currentTime;
    prevAdcValue = adcValue;
  }
  
  stopCharge();
  
  if (isLongMeasurement) {
    finalizeLongMeasurementDisplay();
  }
  
  return result;
}

/**
 * @brief 通常測定（時定数）による容量計算
 * RC時定数から静電容量を算出: C = t / R
 * 測定時間がTARGET_ADC（63.2%）に達するまでの時間を用いて計算
 * @param elapsedTime 測定時間（μs）
 * @return 静電容量（μF）、測定不可能な場合は0
 */
float calculateCapacitanceFromTimeConstant(unsigned long elapsedTime) {
  unsigned long correctedTime = (elapsedTime >= offsetTime) ? (elapsedTime - offsetTime) : 0;
  
  // 測定限界以下の場合は0を返す（analogRead()実行時間2回分より短ければ、測定限界以下とみなす）
  correctedTime = (correctedTime < analogReadTimeUs * 2) ? 0 : correctedTime;
  
  if (correctedTime == 0) return 0;

#ifdef DEBUG
  Serial.print("Charge time: ");
  printWithComma(elapsedTime);
  Serial.print(" us (corrected: ");
  printWithComma(correctedTime);
  Serial.print(" us, offset: ");
  printWithComma(offsetTime);
  Serial.print(" us, analogReadTimeUs: ");
  Serial.print(analogReadTimeUs);
  Serial.print(" us)  ADC after charge: ");
  Serial.print(TARGET_ADC);
#endif

  // 通常測定：時定数から計算
  float capacitance = ((float)correctedTime / 1e6f) / CHARGE_RESISTOR;
  
  return capacitance * 1e6f; // F → μF変換
}

/**
 * @brief タイムアウト時（ADC値）による容量計算
 * タイムアウト時のADC値から静電容量を推定: C = -t / (R × ln(1 - V/Vcc))
 * 大容量コンデンサでタイムアウトした場合の推定計算
 * @param finalADC タイムアウト時のADC値
 * @return 静電容量（μF）、測定不可能な場合は0または-1
 */
float calculateCapacitanceFromADC(int finalADC) {
  if (finalADC <= 0) return 0;
  
  // ADC値が閾値を下回る場合は測定不能
  if (finalADC < MIN_ADC_FOR_MEASUREMENT) {
    return -1; // 測定不能を示す特別な値
  }
  
  float ratio = (float)finalADC / (float)ADC_DIVISIONS;
  
  if (ratio >= 0.999f) return 0; // ほぼ満充電の場合は計算不可
  
  // タイムアウト時間から浮遊容量補正
  unsigned long correctedTime = (CHARGE_TIMEOUT_US >= offsetTime) ? 
                                (CHARGE_TIMEOUT_US - offsetTime) : 0;
  
  // 測定限界以下の場合は0を返す
  correctedTime = (correctedTime < analogReadTimeUs * 2) ? 0 : correctedTime;
  
  if (correctedTime == 0) return 0;

#ifdef DEBUG
  Serial.print("Timeout - ADC: ");
  Serial.print(finalADC);
  Serial.print(", calculated: ");
#endif

  float timeSeconds = (float)correctedTime / 1e6f; // μs → 秒
  float capacitance = -timeSeconds / (CHARGE_RESISTOR * log(1.0f - ratio));
  
  return capacitance * 1e6f; // F → μF変換
}

/**
 * @brief 補正値（オフセット）を取得
 * 浮遊容量などによる測定誤差を補正するためのオフセット時間を測定
 * @return オフセット時間（μs）
 */
unsigned long measureOffset() {
  const int N = 10;
  unsigned long values[N];
  
  for (int i = 0; i < N; i++) {
    ChargeResult result = measureChargeTime();
    
    if (result.elapsedTime == 0) {
      values[i] = 0;
#ifdef DEBUG
      Serial.print("Offset measurement #");
      Serial.print(i + 1);
      Serial.println(": timeout.");
#endif
    } else {
      values[i] = result.elapsedTime;
#ifdef DEBUG
      Serial.print("Offset measurement #");
      Serial.print(i + 1);
      Serial.print(": elapsedTime = ");
      printWithComma(result.elapsedTime);
      Serial.println(" us");
#endif
    }
  }

  // 最大値・最小値を除いた平均値を計算
  unsigned long maxVal = values[0], minVal = values[0], sum = 0;
  for (int i = 0; i < N; i++) {
    if (values[i] > maxVal) maxVal = values[i];
    if (values[i] < minVal) minVal = values[i];
    sum += values[i];
  }
  sum -= maxVal;
  sum -= minVal;
  unsigned long offset = sum / (N - 2);

#ifdef DEBUG
  Serial.print("Offset time (average, max/min excluded): ");
  printWithComma(offset);
  Serial.println(" us");
#endif
  return offset;
}

// =====================================================================
//  結果表示・処理関数群
// =====================================================================
/**
 * @brief 指定値（μF単位）に最適な表示単位のインデックスを返す
 * @param value_uF 容量値（μF単位）
 * @return 単位インデックス
 */
int getDisplayUnitIndex(float value_uF) {
  float absoluteValue = fabs(value_uF);
  if (absoluteValue >= 1e6f) return 0;
  if (absoluteValue >= 1e3f) return 1;
  if (absoluteValue >= 1.0f) return 2;
  if (absoluteValue >= 0.001f) return 3;
  return 4;
}

/**
 * @brief μF単位の値を指定した単位でシリアル表示する
 * @param value_uF 容量値（μF単位）
 * @param unit_idx 単位インデックス
 */
void printValueWithUnit(float value_uF, int unit_idx) {
  float displayValue = value_uF * UNIT_FACTORS[unit_idx];
  int intDigits = (int)(displayValue == 0 ? 1 : floor(log10(fabs(displayValue))) + 1);
  int decimals = max(0, 4 - intDigits);
  if (decimals > 3) decimals = 3;
  Serial.print(displayValue, decimals);
  Serial.print(SI_PREFIX[unit_idx]);
}

/**
 * @brief 入力値から最も近いE12系列値を判定し、LCDとシリアルに結果を表示
 * @param value_uF 測定された容量値（μF単位）
 */
void showE12Result(float value_uF) {
  if (value_uF <= 0) {
    Serial.println(F("値が0以下です。"));
    return;
  }

  // 入力値のデケード指数と正規化値を算出（例: 12.3μF → 1.23×10^1）
  int decadeExp = (int)floor(log10(value_uF));
  float decade = pow(10, decadeExp);
  float normValue = value_uF / decade;

  // E12系列の全値を大きい方から比較する。差が同じ場合は分母が大きいほうがパーセント表示の値が小さくなる
  float minDiff = 1e9f;
  float closestE12 = 0;
  int closestExp = decadeExp;

  for (int seriesIndex = E12_SIZE - 1; seriesIndex >= 0; seriesIndex--) {
    float candidate = E12_SERIES[seriesIndex];
    float diff = fabs(normValue - candidate);
    if (diff < minDiff) {
      minDiff = diff;
      closestE12 = candidate;
      closestExp = decadeExp;
    }
  }

  // 正規化値がE12最大値より大きい場合は、次のデケードの最小値（E12_SERIES[0]）とも比較する
  float diffNext = fabs(normValue - (E12_SERIES[0] * 10.0f));

  if (diffNext < minDiff) {
    minDiff = diffNext;
    closestE12 = E12_SERIES[0];
    closestExp = decadeExp + 1;
  }

  // 正規化値がE12最小値より小さい場合は、前のデケードの最大値（E12_SERIES[E12_SIZE-1]）とも比較する
  float diffPrev = fabs(normValue - (E12_SERIES[E12_SIZE - 1] * 0.1f));
  
  if (diffPrev < minDiff) {
    minDiff = diffPrev;
    closestE12 = E12_SERIES[E12_SIZE - 1];
    closestExp = decadeExp - 1;
  }

  // 最も近いE12値（μF単位）と差分・誤差率を算出
  float closestValue = closestE12 * pow(10, closestExp);
  float diffAbs = value_uF - closestValue;
  float diffPct = (diffAbs / closestValue) * 100.0f;

  // E12値の単位インデックスを決定（表示単位をE12値に合わせる）
  int unit_idx = getDisplayUnitIndex(closestValue);
  
  // LCD表示用の文字列を構築
  char buf1[10], buf2[8], buf3[8], buf4[8];
  formatCapacitanceForLCD(buf1, value_uF, unit_idx);      // 測定値
  formatDifferenceForLCD(buf2, diffAbs, unit_idx);        // 差分
  formatE12ValueForLCD(buf3, closestValue, unit_idx);     // E12値
  formatPercentForLCD(buf4, diffPct);                     // 誤差率（%）

  char lcd_line[18];
  snprintf(lcd_line, 17, "%-8s  %-6s", buf1, buf2);     // LCD 1行目：測定値   差分
  lcd.setCursor(0, 0);
  lcd.print(lcd_line);
  
  snprintf(lcd_line, 17, "%-5s E12 %-6s", buf3, buf4);  // LCD 2行目：E12値 E12  誤差率%
  lcd.setCursor(0, 1);
  lcd.print(lcd_line);

  // シリアル出力（測定値、E12値、差分、誤差率）
  printValueWithUnit(value_uF, unit_idx);
  Serial.print("F  E12: ");
  printValueWithUnit(closestValue, unit_idx);
  Serial.print("F  Δ: ");
  if (diffAbs > 0) Serial.print("+");
  printValueWithUnit(diffAbs, unit_idx);
  Serial.print("F (");
  if (diffPct > 0) Serial.print("+");
  Serial.print(diffPct, 2);
  Serial.println("%)");
}

// =====================================================================
//  UI・状態管理関数
// =====================================================================
/**
 * @brief （起動時に）測定端子が開放されているかをチェックし、コンデンサが接続されているなら停止する
 */
void checkOpenTerminal() {
  discharge(DEFAULT_DISCHARGE_TIME);

  unsigned long startTime = startCharge();
  unsigned long elapsed = 0;

  while (analogRead(ANALOG_PIN) < TARGET_ADC) {
    elapsed = micros() - startTime;
    if (elapsed > MIN_CHARGE_TIME_US) { // 所定時間経ってもADCが所定値に達しないならコンデンサがつながっていると判断（エラー）
      Serial.println("Please disconnect the capacitor and restart the device.");
      lcd.setCursor(0, 0);
      lcd.print("Disconnect C!   ");
      lcd.setCursor(0, 1);
      lcd.print("and restart     ");
      while (true) {  // 無限ループで停止
        delay(1000);
      }
    }
  }

  stopCharge();
}

/**
 * @brief コンデンサが外されるまで待機する
 * ADCが所定の電圧を超えたら（大容量）コンデンサが外された（または、ショートが解消した）と判断
 * @param wait_ms 待機間隔（ms）
 */
void waitForCapacitorRemoval(unsigned long wait_ms) {
  static unsigned long prevBlink = 0;
  static bool ledState = false;

  startCharge();

  while (analogRead(ANALOG_PIN) < TARGET_ADC) {
    // LED点滅（1秒ごと）
    unsigned long now = millis();
    if (now - prevBlink >= 1000) {
      ledState = !ledState;
      digitalWrite(LED_PIN, ledState ? HIGH : LOW);
      prevBlink = now;
    }
    delay(wait_ms);
  }

  stopCharge();
  discharge(DEFAULT_DISCHARGE_TIME);  // 念のため放電
}

// =====================================================================
//  メインプログラム
// =====================================================================
/**
 * @brief ピンの初期化処理
 */
void initializePins() {
  pinMode(PULSE_PIN, OUTPUT);
  digitalWrite(PULSE_PIN, LOW);

  pinMode(DISCHARGE_PIN, OUTPUT);
  digitalWrite(DISCHARGE_PIN, LOW);

  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LOW);
}

/**
 * @brief シリアル通信初期化待ち（プログレスバー表示付き）
 * シリアル通信の安定化待ち時間中にプログレスバーを表示
 * @param duration_ms 待ち時間（ミリ秒）
 */
void waitSerialWithProgress(int duration_ms) {
  const int LCD_COLS = 16;
  int delay_per_step = duration_ms / LCD_COLS;

  lcd.setCursor(0, 0);
  lcd.print("Initializing... ");
  lcd.setCursor(0, 1);

  for (int i = 0; i < LCD_COLS; i++) {
    lcd.setCursor(i, 1);
    lcd.print("#");
    delay(delay_per_step);
  }
}

/**
 * @brief セットアップ関数
 * システム初期化、ピン設定、オフセット測定を実行
 */
void setup() {
  Serial.begin(115200); // シリアル通信速度を115200bpsに設定

  initializePins();
  lcd.init();
  lcd.backlight();
  lcd.clear();

  waitSerialWithProgress(2000);   // シリアル通信の安定化のため短い待ち時間を追加
  
  // 起動メッセージ
  Serial.println("Initializing... ");

  checkOpenTerminal();

  // analogRead()の平均処理時間を実測
  analogReadTimeUs = measureAnalogReadTime();

#ifdef DEBUG
  Serial.print("analogRead() average time: ");
  Serial.print(analogReadTimeUs);
  Serial.println(" us");
  Serial.println("Measuring offset (leave input open)...");
#endif

  offsetTime = measureOffset();

#ifdef DEBUG
  Serial.print("Offset time (final): ");
  printWithComma(offsetTime);
  Serial.println(" us");
#endif

  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print("Cap Meter Ready.");

  Serial.println("Capacitance Meter with E12 Series Comparison Ready.");
}

/**
 * @brief メインループ関数
 * 1秒間隔で容量測定を実行し、結果をE12系列と比較して表示
 */
void loop() {
  static unsigned long prevBlink = 0;
  static bool ledState = false;
  unsigned long now = millis();

  // LED点滅処理（1秒ごと）
  if (now - prevBlink >= 1000) {
    ledState = !ledState;
    digitalWrite(LED_PIN, ledState ? HIGH : LOW);
    prevBlink = now;
  }

  unsigned long measureStartMillis = millis();
  float capacitance_uF = -1.0f; // 無効値で初期化

#ifdef DEBUG
  Serial.println("Measuring...");
#endif

  ChargeResult result = measureChargeTime();
  
  if (result.elapsedTime == 0) {
    // タイムアウト時：ADC値と時間（タイムアウト時間）算出
    capacitance_uF = calculateCapacitanceFromADC(result.finalADC);
    if (capacitance_uF <= 0) {
      Serial.println("Capacitance too large or terminals shorted (measurement impossible).");
      lcd.setCursor(0, 0);
      lcd.print("No capacitor    ");
      lcd.setCursor(0, 1);
      lcd.print("Too large/Short ");
      currentScreenState = NO_CAPACITOR;  // 状態を更新
      waitForCapacitorRemoval(100);
      return; // 待機処理後、次のループの先頭に戻る
    } else {
#ifdef DEBUG
      Serial.println(); // デバッグメッセージの改行
#endif
      Serial.print("Timeout calculation: ");
    }
  } else {
    // 通常測定：時定数から計算
    capacitance_uF = calculateCapacitanceFromTimeConstant(result.elapsedTime);
    if (capacitance_uF <= 0) {
      Serial.println("Measurement below detection limit.");
      lcd.setCursor(0, 0);
      lcd.print("No capacitor    ");
      lcd.setCursor(0, 1);
      lcd.print("Open terminal   ");
      currentScreenState = NO_CAPACITOR;  // 状態を更新
    } else {
#ifdef DEBUG
      Serial.println(); // デバッグメッセージの改行
#endif
      Serial.print("Normal measurement: ");
    }
  }

  // 有効な測定結果があればE12系列判定・表示
  if (capacitance_uF > 0) {
    showE12Result(capacitance_uF);
    currentScreenState = MEASUREMENT_RESULT;  // 状態を更新
  }

  // 1秒間隔処理
  unsigned long measureElapsed = millis() - measureStartMillis;
  if (measureElapsed < 1000) {
    delay(1000 - measureElapsed);
  }
}
