/*
 * Arduino RC時定数式コンデンサ容量計 with E12系列判定
 * 抵抗RとコンデンサCを直列に接続し、アナログ値が基準値の63.2%に達するまでの時間を計測
 * 測定結果をE12系列と比較してシリアルモニタに表示
 *
 * 測定時間1秒未満で測定できる容量範囲（目安、ただし1MΩ以外は実機未確認）:
 *   CHARGE_RESISTOR = 1MΩ    : 約1nF ～ 1μF
 *   CHARGE_RESISTOR = 100kΩ  : 約10nF ～ 10μF
 *   CHARGE_RESISTOR = 1kΩ    : 約1μF ～ 1000μF（1mF）
 *
 * © JH4VAJ 2025
 */

// #define DEBUG                  // デバッグ表示を有効にする

// === 測定パラメータ ===
#define CHARGE_TIMEOUT_US         3000000 // 充電タイムアウト（μs）3秒
#define MIN_ADC_FOR_MEASUREMENT   200     // タイムアウト時の測定可能最小ADC値
#define ADC_SAMPLE_HOLD_DELAY_US  12      // ADCサンプル&ホールド遅延補正（μs）
                                          // Arduino ADCクロック125kHzでの1.5クロック分（1.5/125kHz = 12μs）
#define DEFAULT_DISCHARGE_TIME    100     // 放電時間（ms）
                                          // 【例】放電用抵抗220Ω・コンデンサ10μFの場合、99.9%放電にかかる時間は約15.2ms

// === ハードウェア設定 ===
const int PULSE_PIN = 2;      // 充電用デジタルピン
const int DISCHARGE_PIN = 3;  // 放電用デジタルピン（推奨：220Ω抵抗を介してコンデンサに接続）
const int ANALOG_PIN = A1;    // コンデンサ電圧測定ピン
const int LED_PIN = LED_BUILTIN; // LED点滅用ピン（通常はD13）

// === 回路定数 ===
const float CHARGE_RESISTOR = 1000000.0;   // 充電抵抗値（Ω）※実測値に合わせて調整すること（例：1MΩ = 1,000,000Ω）
const int TARGET_ADC = (int)(1024 * 0.632); // 基準値の63.2%（約647）
const int ADC_DIVISIONS = 1024;           // ADC分解能

// === E12系列定義 ===
const float E12_SERIES[] = {1.0, 1.2, 1.5, 1.8, 2.2, 2.7, 3.3, 3.9, 4.7, 5.6, 6.8, 8.2};
const int E12_SIZE = sizeof(E12_SERIES) / sizeof(E12_SERIES[0]);

// === 単位変換定義 ===
const char* UNIT_LABELS[] = {"F", "mF", "uF", "nF", "pF"};
const float UNIT_FACTORS[] = {1e-6, 1e-3, 1.0, 1e3, 1e6}; // F, mF, μF, nF, pF

// === グローバル変数 ===
unsigned long offsetTime = 0; // 浮遊容量補正値（μs）
unsigned long analogReadTimeUs = 100; // analogRead()の平均実測時間（μs）

/**
 * 測定結果を格納する構造体
 */
struct ChargeResult {
  unsigned long elapsedTime;  // 測定時間（μs）、タイムアウト時は0
  int finalADC;              // 最終ADC値
};

/**
 * analogRead()の平均処理時間（μs）を測定する関数
 * 10回連続で測定して平均値を算出
 * @return analogRead()の平均処理時間（μs）
 */
unsigned long measureAnalogReadTime() {
  analogRead(ANALOG_PIN);   // 安定化のため空読み（念のため、2回）
  analogRead(ANALOG_PIN);

  unsigned long t1 = micros();
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  analogRead(ANALOG_PIN);
  unsigned long t2 = micros();

  return (t2 - t1) / 10;
}

/**
 * 充電時間測定関数
 * コンデンサを放電後、充電してTARGET_ADCに達するまでの時間を測定
 * @return ChargeResult構造体（測定時間とADC値）
 */
ChargeResult measureChargeTime() {
  ChargeResult result = {0, 0};
  
  // 放電処理
#ifdef DEBUG
  Serial.print("ADC before discharge: ");
  Serial.print(analogRead(ANALOG_PIN));
#endif

  pinMode(DISCHARGE_PIN, OUTPUT);
  digitalWrite(DISCHARGE_PIN, LOW);
  delay(DEFAULT_DISCHARGE_TIME);
  pinMode(DISCHARGE_PIN, INPUT); // 放電完了後、ピンをハイインピーダンス状態にする
  
#ifdef DEBUG
  Serial.print("  ADC after discharge: ");
  Serial.println(analogRead(ANALOG_PIN));
#endif

  // 充電開始
  digitalWrite(PULSE_PIN, HIGH);
  unsigned long startTime = micros();
  
  unsigned long prevTime = startTime;
  int prevAdcValue = 0;

  while (true) {
    unsigned long currentTime = micros();
    int adcValue = analogRead(ANALOG_PIN);
    
    if (adcValue >= TARGET_ADC) {
      // 線形補間でより正確な時刻を推定
      float ratio = (float)(TARGET_ADC - prevAdcValue) / (float)(adcValue - prevAdcValue);
      unsigned long interpolatedTime = prevTime + (unsigned long)((currentTime - prevTime) * ratio);
      result.elapsedTime = interpolatedTime - startTime;
      
      result.elapsedTime += ADC_SAMPLE_HOLD_DELAY_US;  // ADCサンプル&ホールド遅延補正
      result.finalADC = TARGET_ADC;
      break;
    }
    
    if (currentTime - startTime > CHARGE_TIMEOUT_US) {
      result.elapsedTime = 0; // タイムアウト時は0を設定
      result.finalADC = adcValue;
      break;
    }
    
    // 線形補間計算のため前回の測定値を保存
    prevTime = currentTime;
    prevAdcValue = adcValue;
  }
  
  digitalWrite(PULSE_PIN, LOW); // 充電停止
  
  return result;
}

/**
 * 通常測定（時定数）による容量計算
 * RC時定数から静電容量を算出
 * @param elapsedTime 測定時間（μs）
 * @return 静電容量（μF）、測定不可能な場合は0
 */
float calculateCapacitanceFromTimeConstant(unsigned long elapsedTime) {
  unsigned long correctedTime = (elapsedTime >= offsetTime) ? (elapsedTime - offsetTime) : 0;
  
  // 測定限界以下の場合は0を返す（analogRead()実行時間2回分より短ければ、測定限界以下とみなす）
  correctedTime = (correctedTime < analogReadTimeUs * 2) ? 0 : correctedTime;
  
  if (correctedTime == 0) return 0;

#ifdef DEBUG
  Serial.print("Charge time: ");
  printWithComma(elapsedTime);
  Serial.print(" us (corrected: ");
  printWithComma(correctedTime);
  Serial.print(" us, offset: ");
  printWithComma(offsetTime);
  Serial.print(" us, analogReadTimeUs: ");
  Serial.print(analogReadTimeUs);
  Serial.print(" us)  ADC after charge: ");
  Serial.print(TARGET_ADC);
#endif

  // 通常測定：時定数から計算
  float capacitance = ((float)correctedTime / 1e6) / CHARGE_RESISTOR;
  
  return capacitance * 1e6; // F → μF変換
}

/**
 * タイムアウト時（ADC値）による容量計算
 * ADC値から逆算して静電容量を算出
 * @param finalADC タイムアウト時のADC値
 * @return 静電容量（μF）、測定不可能な場合は0または-1
 */
float calculateCapacitanceFromADC(int finalADC) {
  if (finalADC <= 0) return 0;
  
  // ADC値が閾値を下回る場合は測定不能
  if (finalADC < MIN_ADC_FOR_MEASUREMENT) {
    return -1; // 測定不能を示す特別な値
  }
  
  float ratio = (float)finalADC / (float)ADC_DIVISIONS;
  
  if (ratio >= 0.999) return 0; // ほぼ満充電の場合は計算不可
  
  // タイムアウト時間から浮遊容量補正
  unsigned long correctedTime = (CHARGE_TIMEOUT_US >= offsetTime) ? 
                                (CHARGE_TIMEOUT_US - offsetTime) : 0;
  
  // 測定限界以下の場合は0を返す
  correctedTime = (correctedTime < analogReadTimeUs * 2) ? 0 : correctedTime;
  
  if (correctedTime == 0) return 0;

#ifdef DEBUG
  Serial.print("Timeout - ADC: ");
  Serial.print(finalADC);
  Serial.print(", calculated: ");
#endif

  float timeSeconds = correctedTime / 1e6; // μs → 秒
  float capacitance = -timeSeconds / (CHARGE_RESISTOR * log(1.0 - ratio));
  
  return capacitance * 1e6; // F → μF変換
}

/**
 * 補正値（オフセット）を取得
 * 浮遊容量による測定誤差を補正するためのオフセット時間を測定
 * @return オフセット時間（μs）
 */
unsigned long measureOffset() {
  const int N = 10;
  unsigned long values[N];
  
  for (int i = 0; i < N; i++) {
    ChargeResult result = measureChargeTime();
    
    if (result.elapsedTime == 0) {
      values[i] = 0;
#ifdef DEBUG
      Serial.print("Offset measurement #");
      Serial.print(i + 1);
      Serial.println(": timeout.");
#endif
    } else {
      values[i] = result.elapsedTime;
#ifdef DEBUG
      Serial.print("Offset measurement #");
      Serial.print(i + 1);
      Serial.print(": elapsedTime = ");
      printWithComma(result.elapsedTime);
      Serial.println(" us");
#endif
    }
  }

  // 最大値・最小値を除いた平均値を計算
  unsigned long maxVal = values[0], minVal = values[0], sum = 0;
  for (int i = 0; i < N; i++) {
    if (values[i] > maxVal) maxVal = values[i];
    if (values[i] < minVal) minVal = values[i];
    sum += values[i];
  }
  sum -= maxVal;
  sum -= minVal;
  unsigned long offset = sum / (N - 2);

#ifdef DEBUG
  Serial.print("Offset time (average, max/min excluded): ");
  printWithComma(offset);
  Serial.println(" us");
#endif
  return offset;
}

/**
 * 指定値（μF単位）に最適な表示単位のインデックスを返す
 * @param value_uF 容量値（μF単位）
 * @return 単位インデックス（0=F, 1=mF, 2=μF, 3=nF, 4=pF）
 */
int getUnitIndex(float value_uF) {
  float absoluteValue = abs(value_uF);
  if (absoluteValue >= 1e6) return 0; // F
  if (absoluteValue >= 1e3) return 1; // mF
  if (absoluteValue >= 1.0) return 2; // μF
  if (absoluteValue >= 0.001) return 3; // nF
  return 4; // pF
}

/**
 * μF単位の値を指定した単位で表示する
 * @param value_uF 容量値（μF単位）
 * @param unit_idx 単位インデックス（0=F, 1=mF, 2=μF, 3=nF, 4=pF）
 */
void printWithUnit(float value_uF, int unit_idx) {
  float displayValue = value_uF * UNIT_FACTORS[unit_idx];
  Serial.print(displayValue, 3);
  Serial.print(" ");
  Serial.print(UNIT_LABELS[unit_idx]);
}

/**
 * 入力値（μF単位）から最も近いE12系列値を判定し結果を表示
 * E12値の単位に合わせて入力値・E12値・差分を同じ単位で1行表示
 * @param value_uF 測定された容量値（μF単位）
 */
void processAndPrintE12Result(float value_uF) {
  if (value_uF <= 0) {
    Serial.println(F("値が0以下です。"));
    return;
  }

  // 入力値のデケード指数と正規化値を算出
  int decadeExp = (int)floor(log10(value_uF));
  float decade = pow(10, decadeExp);
  float normValue = value_uF / decade;

  // E12系列の全値を大きい方から比較する。差が同じ場合は分母が大きいほうがパーセント表示の値が小さくなる
  float minDiff = 1e9;
  float closestE12 = 0;
  int closestExp = decadeExp;
  for (int seriesIndex = E12_SIZE - 1; seriesIndex >= 0; seriesIndex--) {
    float candidate = E12_SERIES[seriesIndex];
    float diff = abs(normValue - candidate);
    if (diff < minDiff) {
      minDiff = diff;
      closestE12 = candidate;
      closestExp = decadeExp;
    }
  }
  // 正規化値がE12最大値より大きい場合は、次のデケードの最小値（E12_SERIES[0]）とも比較する
  float diffNext = abs(normValue - (E12_SERIES[0] * 10.0));
  if (diffNext < minDiff) {
    minDiff = diffNext;
    closestE12 = E12_SERIES[0];
    closestExp = decadeExp + 1;
  }
  // 正規化値がE12最小値より小さい場合は、前のデケードの最大値（E12_SERIES[E12_SIZE-1]）とも比較する
  float diffPrev = abs(normValue - (E12_SERIES[E12_SIZE - 1] * 0.1));
  if (diffPrev < minDiff) {
    minDiff = diffPrev;
    closestE12 = E12_SERIES[E12_SIZE - 1];
    closestExp = decadeExp - 1;
  }

  float closestValue = closestE12 * pow(10, closestExp);
  float diffAbs = value_uF - closestValue;
  float diffPct = (diffAbs / closestValue) * 100.0;

  // E12値の単位インデックスを決定（表示単位をE12値に合わせる）
  int unit_idx = getUnitIndex(closestValue);

  // 結果を1行で表示（E12系列値の単位で統一）
  Serial.print("Measured: ");
  printWithUnit(value_uF, unit_idx);
  Serial.print("  E12: ");
  printWithUnit(closestValue, unit_idx);
  Serial.print("  Δ: ");
  printWithUnit(diffAbs, unit_idx);
  Serial.print(" (");
  Serial.print(diffPct, 2);
  Serial.println("%)");
}

/**
 * ピンの初期化処理
 */
void initializePins() {
  pinMode(PULSE_PIN, OUTPUT);
  pinMode(DISCHARGE_PIN, OUTPUT);
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(PULSE_PIN, LOW);
  digitalWrite(DISCHARGE_PIN, LOW);
  digitalWrite(LED_PIN, LOW);
}

/**
 * 桁区切り付きでunsigned long値をシリアルに出力する関数
 * @param num 出力する数値
 */
void printWithComma(unsigned long num) {
  char buf[20];
  int len = 0;
  int count = 0;
  if (num == 0) {
    Serial.print("0");
    return;
  }
  // 逆順に3桁ごとにカンマ挿入
  while (num > 0) {
    if (count && count % 3 == 0) buf[len++] = ',';
    buf[len++] = '0' + (num % 10);
    num /= 10;
    count++;
  }
  // 逆順なので正順に出力
  for (int i = len - 1; i >= 0; i--) {
    Serial.print(buf[i]);
  }
}

/**
 * セットアップ関数
 * システム初期化、ピン設定、オフセット測定を実行
 */
void setup() {
  Serial.begin(115200); // シリアル通信速度を115200bpsに設定
  delay(1000);           // シリアル通信の安定化のため短い待ち時間を追加
  
  // 起動メッセージ
  Serial.println("Capacitance Meter with E12 Series Comparison Ready.");
  
  initializePins();

  // analogRead()の平均処理時間を実測
  analogReadTimeUs = measureAnalogReadTime();
#ifdef DEBUG
  Serial.print("analogRead() average time: ");
  Serial.print(analogReadTimeUs);
  Serial.println(" us");
  Serial.println("Measuring offset (leave input open)...");
#endif

  offsetTime = measureOffset();
}

/**
 * メインループ関数
 * 1秒間隔で容量測定を実行し、結果をE12系列と比較して表示
 */
void loop() {
  static unsigned long prevBlink = 0;
  static bool ledState = false;
  unsigned long now = millis();

  // LED点滅処理（1秒ごと）
  if (now - prevBlink >= 1000) {
    ledState = !ledState;
    digitalWrite(LED_PIN, ledState ? HIGH : LOW);
    prevBlink = now;
  }

  unsigned long measureStartMillis = millis();
  float c_uF = -1.0; // 無効値で初期化

#ifdef DEBUG
  Serial.println("Measuring...");
#endif

  ChargeResult result = measureChargeTime();
  
  if (result.elapsedTime == 0) {
    // タイムアウト時：ADC値から逆算
    c_uF = calculateCapacitanceFromADC(result.finalADC);
    if (c_uF <= 0) {
      Serial.println("Capacitance too large or terminals shorted (measurement impossible).");
    } else {
#ifdef DEBUG
      Serial.println(); // デバッグメッセージの改行
#endif
      Serial.print("Timeout calculation: ");
    }
  } else {
    // 通常測定：時定数から計算
    c_uF = calculateCapacitanceFromTimeConstant(result.elapsedTime);
    if (c_uF <= 0) {
      Serial.println("Measurement below detection limit.");
    } else {
#ifdef DEBUG
      Serial.println(); // デバッグメッセージの改行
#endif
      Serial.print("Normal measurement: ");
    }
  }

  // 有効な測定結果があればE12系列判定・表示
  if (c_uF > 0) {
    processAndPrintE12Result(c_uF);
  }

  // 1秒間隔処理
  unsigned long measureElapsed = millis() - measureStartMillis;
  if (measureElapsed < 1000) {
    delay(1000 - measureElapsed);
  }
}
