# worker.py
import uasyncio as asyncio
import struct, os, time
from protocol import *
from rs485stream import RS485Stream
import raytracer as rt
from gpio_chain import ChainIO

# --- ワーカーIDを id.txt から読む ---
def read_worker_id(path="id.txt", default=1):
    try:
        with open(path, "r") as f:
            n = int(f.read().strip())
            if 1 <= n <= 254:
                print("worker id:", n)
                return n
    except Exception as e:
        print("id.txt read failed:", e)
    print("fallback id:", default)
    return default

MY_ADDR = read_worker_id()

# 共有状態
have_job  = False
done_flag = False
imgW = 240; imgH = 240
x0 = y0 = w = h = 0
tile_path = None

def ensure_free_space(required_bytes):
    try:
        st = os.statvfs("/")
        free = st[0] * st[3]
        return free >= required_bytes + 8192
    except:
        return True

async def render_to_flash():
    global done_flag, tile_path
    tile_path = "/tile_%02d.bin" % MY_ADDR
    try:
        os.remove(tile_path)
    except OSError:
        pass
    total = 2 * w * h
    if not ensure_free_space(total):
        raise MemoryError("flash free space too small")
    with open(tile_path, "wb") as f:
        for line in range(h):
            print(f"worker[{MY_ADDR}] rendering {line+1}/{h}")
            y_lcd = y0 + line
            data = rt.line_rgb565_for_lcd(x0, y_lcd, w)
            f.write(data)
            await asyncio.sleep_ms(0)
    done_flag = True

def iter_lines_from_flash():
    line_bytes = 2 * w
    mv = bytearray(line_bytes)
    mvv = memoryview(mv)
    with open(tile_path, "rb") as f:
        for _ in range(h):
            n = f.readinto(mvv)
            if n != line_bytes:
                raise OSError("short read")
            yield mvv

async def rx_loop(bus: RS485Stream):
    global have_job, done_flag, imgW, imgH, x0, y0, w, h, tile_path
    print(f"worker[{MY_ADDR}@RXLOOP] task rx_loop start")
    while True:
        fr = await bus.recv_frame(max_len=64)
        if fr is None:
            print(f"worker[{MY_ADDR}@RXLOOP] recv_frame -> fr = None")
            await asyncio.sleep_ms(0)
            continue
        addr, cmd, payload = fr
        print(f"worker[{MY_ADDR}@RXLOOP] recv_frame -> addr = {addr}, cmd = {cmd}, payload = {payload}")
        # 宛先=自分宛のみ受理（CMD_TOKENはassignedのみが受ける設計）
        if addr != MY_ADDR:
            continue

        if cmd == CMD_ASSIGN and len(payload) == 1 + 12:
            print(f"worker[{MY_ADDR}@RXLOOP] receive CMD_ASSIGN")
            seq = payload[0]
            imgW, imgH, x0, y0, w, h = struct.unpack("<HHHHHH", payload[1:1+12])
            have_job  = True
            done_flag = False
            try:
                if tile_path: os.remove(tile_path)
            except OSError:
                pass
            print(f"worker[{MY_ADDR}@RXLOOP] send ACK")
            await bus.send_frame(addr=addr, cmd=CMD_ACK, payload=bytes([seq]))

        elif cmd == CMD_TOKEN:
            print(f"worker[{MY_ADDR}@RXLOOP] receive CMD_TOKEN")
            ids = list(payload)
            if not ids or ids[0] != MY_ADDR:
                continue
            # 自分の番：タイル送信
            if not (have_job and done_flag and w > 0 and h > 0):
                # 一応ガード
                continue
            header = struct.pack("<HHHH", x0, y0, w, h)
            total_len = 8 + 2*w*h
            # 送出直前ドレイン（安全）
            n = bus.uart.any()
            if n: bus.uart.read(n)
            bus.send_frame_streaming_sync(MASTER_ADDR, CMD_TILE, total_len,
                                          header=header,
                                          chunk_iter=iter_lines_from_flash(),
                                          interline_gap_ms=2)
            # 次ノードへトークン
            rest = ids[1:]
            if rest:
                await bus.send_frame(rest[0], CMD_TOKEN, bytes(rest))
            # タイルは保持しても良いが、次フレームに備えて消しておく
            try:
                if tile_path: os.remove(tile_path)
            except OSError:
                pass
            have_job  = False
            done_flag = False

async def gpio_sync_loop(chain: ChainIO):
    """
    GPIOリング同期：未割当ノードは即時中継、割当ノードは開始でHigh, 完了でLow
    """
    global have_job, done_flag
    print(f"worker[{MY_ADDR}] task gpio_sync_loop start")
    while True:
        # Start波待ち
        print(f"worker[{MY_ADDR}@GPIO] call chain.wait_in_level: wait for 1 (wait rendering START)")
        ok = await chain.wait_in_level(1, 3600000)  # 長め（運用に応じて）
        if not ok:
            await asyncio.sleep_ms(10)
            continue

        if have_job:
            print(f"worker[{MY_ADDR}@GPIO] set gpio HIGH and start rendering")
            chain.drive_high()
            # レンダ開始
            print(f"worker[{MY_ADDR}@GPIO] START rendering")
            await render_to_flash()
            print(f"worker[{MY_ADDR}@GPIO] END rendering")
            print(f"worker[{MY_ADDR}@GPIO] set gpio HIGH")
            print(f"worker[{MY_ADDR}@GPIO] -> OK")
        else:
            # 未割当：即中継
            print(f"worker[{MY_ADDR}] set gpio HIGH and go through")
            chain.drive_high()

        # Done波（Low）待ち
        print(f"worker[{MY_ADDR}@GPIO] call chain.wait_in_level: wait for 0 (wait rendering STOP")
        ok = await chain.wait_in_level(0, 3600000)
        if not ok:
            # 異常時は落とす
            chain.drive_low()
            continue

        if have_job:
            # 完了済みのはず
            print(f"worker[{MY_ADDR}@GPIO] set gpio LOW")
            chain.drive_low()
        else:
            # 未割当：即中継
            print(f"worker[{MY_ADDR}@GPIO] set gpio LOW")
            chain.drive_low()

async def start():
    bus = RS485Stream(uart_id=0, tx=16, rx=17, dir_pin=18, baudrate=100_000)
    chain = ChainIO(in_pin=20, out_pin=21)  # 初期Low

    await asyncio.gather(
        rx_loop(bus),
        gpio_sync_loop(chain),
    )

def main():
    try:
        asyncio.run(start())
    finally:
        asyncio.new_event_loop()

if __name__ == "__main__":
    main()

