# gpio_chain.py
import uasyncio as asyncio
from machine import Pin
import time

def _deadline_ms(timeout_ms):
    """timeout_ms（Noneは無限）から締切tick値を計算する。"""
    if timeout_ms is None:
        # 十分先の時刻を返す（約12日先）
        return time.ticks_add(time.ticks_ms(), 1_000_000_000)
    return time.ticks_add(time.ticks_ms(), int(timeout_ms))

class ChainIO:
    """
    GPIOリングの1区間を担当するヘルパ
    in_pin:  上流から来る信号を受ける入力（例: GPIO20）
    out_pin: 下流へ出す信号を駆動する出力（例: GPIO21）
    pull:    入力プル設定（既定はプルダウン）
    """
    def __init__(self, in_pin=20, out_pin=21, pull=Pin.PULL_DOWN):
        self.pin_in  = Pin(in_pin, Pin.IN, pull)
        self.pin_out = Pin(out_pin, Pin.OUT, value=0) # 初期値=Low

    # --- 出力制御 ---
    def drive_high(self):
        """出力をHighにする。"""
        print(f"ChainIO: drive_high")
        self.pin_out.value(1)

    def drive_low(self):
        """出力をLowにする。"""
        print(f"ChainIO: drive_low")
        self.pin_out.value(0)

    # --- 非同期待機API ---
    async def wait_in_level(self, level: int, timeout_ms: int | None):
        """
        入力が指定レベル（0/1）になるまで待機する(非ブロッキング)
        True=達成, False=タイムアウト
        """
        print(f"ChainIO: start wait_in_level")
        deadline = _deadline_ms(timeout_ms)
        tgt = 1 if level else 0
        while time.ticks_diff(deadline, time.ticks_ms()) > 0:
            if self.pin_in.value() == tgt:
                print(f"detect pin level: {tgt}")
                return True
            # 1ms刻みで協調的に他タスクへ制御を返す
            await asyncio.sleep_ms(1)
        return False

    async def wait_for_rising(self, timeout_ms: int | None):
        """
        立上りエッジ（0→1）を待つ。True=検出, False=タイムアウト
        """
        deadline = _deadline_ms(timeout_ms)
        prev = self.pin_in.value()
        while time.ticks_diff(deadline, time.ticks_ms()) > 0:
            cur = self.pin_in.value()
            if prev == 0 and cur == 1:
                return True
            prev = cur
            await asyncio.sleep_ms(1)
        return False
